unit UCCDZoomPanel;

interface

uses
  Windows, Graphics, Classes,
  Controls, ExtCtrls, UCCDTypes;

type
  TZoomLevel = 1..4;
  TCCDZoomPanel = class(TGraphicControl)
  private
    FTimer: TTimer;
    FActive: boolean;
    FAboutCCD: TAboutCCD;
    FZoomLevel: TZoomLevel;
    FCrosshairColor: TColor;
    FShowCrosshair: boolean;
    FCrosshairWidth: integer;
    procedure SetActive(value: boolean);
    procedure ZoomTimer(sender: TObject);
  public
    constructor Create(aOwner: TComponent); override;
    destructor Destroy(); override;
  published
    property AboutCCD: TAboutCCD read FAboutCCD stored false;
    property Active: boolean read FActive write SetActive default true;
    property ZoomLevel: TZoomLevel read FZoomLevel write FZoomLevel default 1;
    property CrosshairWidth: integer read FCrosshairWidth write FCrosshairWidth default 1;
    property ShowCrosshair: boolean read FShowCrosshair write FShowCrosshair default true;
    property CrosshairColor: TColor read FCrosshairColor write FCrosshairColor default clBlack;
  end;

implementation

{ TCCDZoomPanel }

constructor TCCDZoomPanel.Create(aOwner: TComponent);
begin
  inherited Create(aOwner);
  FZoomLevel := 1;
  FActive := true;
  FCrosshairWidth := 1;
  FShowCrosshair := true;
  FCrosshairColor := clBlack;
  FTimer := TTimer.Create(self);
  with FTimer do begin
    Interval := 80;
    Enabled := FActive;
    OnTimer := ZoomTimer;
  end;
  self.Width := 200;
  self.Height := 200;
end;

destructor TCCDZoomPanel.Destroy();
begin
  FTimer.Free();
  inherited Destroy();
end;

procedure TCCDZoomPanel.SetActive(value: boolean);
begin
  if (FActive <> value) then begin
    FActive := value;
    FTimer.Enabled := value;
  end;
end;

{ The main algorithm this component (just bellow) is written for Zarco Gajic
  on Delphi.About (http://delphi.about.com/od/graphics/l/aa120198.htm) Thanks!

  We use "GetSysTemMetrics" instead of variable Screen of Forms.pas unit to
  obtain the width and height of monitor resolution and use a particular Pen
  to the optional crosshair via component property.
}
procedure TCCDZoomPanel.ZoomTimer(sender: TObject);
var
  cnv: TCanvas;
  hDesktop: Hwnd;
  cursorPos: TPoint;
  iTmpX, iTmpY: Real;
  sRect, dRect: TRect;
  sWidth, sHeight, iWidth, iHeight, DmX, DmY: integer;
begin
  iWidth := self.Width;
  iHeight := self.Height;
  GetCursorPos(cursorPos);
  hDesktop:= GetDesktopWindow();
  dRect := Rect(0, 0, iWidth, iHeight);
  iTmpX := iWidth / (FZoomLevel * 4);
  iTmpY := iHeight / (FZoomLevel * 4);
  sRect := Rect(cursorPos.x, cursorPos.y, cursorPos.x, cursorPos.y);
  sWidth := GetSysTemMetrics(SM_CXSCREEN);
  sHeight := GetSysTemMetrics(SM_CYSCREEN);
  InflateRect(sRect, Round(iTmpX), Round(iTmpY));
  // move sRect if outside visible area of the screen
  if (sRect.Left < 0) then OffsetRect(sRect, -Srect.Left, 0);
  if (sRect.Top < 0) then OffsetRect(sRect, 0, -Srect.Top);
  if (sRect.Right > sWidth) then
    OffsetRect(sRect, -(sRect.Right-sWidth), 0);
  if (sRect.Bottom > sHeight) then
    OffsetRect(sRect, 0, -(sRect.Bottom-sHeight));
  cnv := TCanvas.Create();
  try
    cnv.Handle := GetDC(hDesktop);
    self.Canvas.CopyRect(dRect,cnv,sRect);
  finally
    ReleaseDC(hDesktop, cnv.Handle);
    cnv.Free();
  end;
  if FShowCrosshair then begin
    with self.Canvas do begin
      Pen.Width := FCrosshairWidth;
      Pen.Color := FCrosshairColor;
      DmX:= (FZoomLevel * 2) * (cursorPos.X-Srect.Left);
      DmY:= (FZoomLevel * 2) * (cursorPos.Y-Srect.Top);
      MoveTo(DmX - (iWidth div 10), DmY); // -
      LineTo(DmX + (iWidth div 10), DmY); // -
      MoveTo(DmX,DmY - (iHeight div 10)); // |
      LineTo(DmX,DmY + (iHeight div 10)); // |
    end;
  end;
end;

end.
