unit UCCDSHFolders;

interface

uses
  Windows, Classes, UCCDBaseComp;

type
  TSHFolderID = (
    fiAdmintools, fiAltstartup, fiAppdata, fiBitbucket, fiCdburnArea,
    fiCommonAdmintools, fiCommonAltstartup, fiCommonAppdata,
    fiCommonDesktopdirectory, fiCommonDocuments, fiCommonFavorites,
    fiCommonMusic, fiCommonPictures, fiCommonPrograms, fiCommonStartmenu,
    fiCommonStartup, fiCommonTemplates, fiCommonVideo, fiComputersnearme,
    fiConnections, fiControls, fiCookies, fiDesktop, fiDesktopdirectory,
    fiDrives, fiFavorites, fiFonts, fiHistory, fiInternet, fiInternetCache,
    fiLocalAppdata, fiMydocuments, fiMymusic, fiMypictures, fiMyvideo,
    fiNethood, fiNetwork, fiPersonal, fiPhotoalbums, fiPlaylists, fiPrinters,
    fiPrinthood, fiProfile, fiProgramFiles, fiProgramFilesCommon, fiPrograms,
    fiRecent, fiResources, fiSampleMusic, fiSamplePlaylists, fiSamplePictures,
    fiSampleVideos, fiSendto, fiStartmenu, fiStartup, fiSystem, fiTemplates,
    fiWindows
  );

type
  TCCDSHFolders = class(TCCDBaseComp)
  private
    FSHFolderPath: string;
    FSHFolderID: TSHFolderID;
    procedure SetSHFolderID(value: TSHFolderID);
    function SHFolderIdToCSIDL(shFolderID: TSHFolderID): integer;
  public
    constructor Create(aOwner: TComponent); override;
    function GetSHFolder(shFolderID: TSHFolderID): string;
  published
    property SHFolderPath: string read FSHFolderPath;
    property SHFolderID: TSHFolderID read FSHFolderID write SetSHFolderID;
  end;

function SHGetSpecialFolderPath(Handle: HWND; Path:
 PChar; Folder: Integer; Create: Bool): BOOL; stdcall;

implementation

uses
  SysUtils;

const
 CSIDL_FLAG_CREATE = $8000;
 { Version 5.0. Combine this CSIDL with any of the following CSIDLs
   to force the creation of the associated folder. }

 CSIDL_ADMINTOOLS = $0030;
 { Version 5.0. The file system directory that is used to store
   administrative tools for an individual user. The Microsoft Management
   Console (MMC) will save customized consoles to this directory, and it
   will roam with the user. }

 CSIDL_ALTSTARTUP = $001d;
 { The file system directory that corresponds to the user's nonlocalized
   Startup program group. }

 CSIDL_APPDATA = $001a;
 { Version 4.71. The file system directory that serves as a common
   repository for application-specific data. A typical path is C:\Documents
   and Settings\username\Application Data. This CSIDL is supported by the
   redistributable Shfolder.dll for systems that do not have the Microsoft
   Internet Explorer 4.0 integrated Shell installed. }

 CSIDL_BITBUCKET = $000a;
 { The virtual folder containing the objects in the user's Recycle Bin. }

 CSIDL_CDBURN_AREA = $003b;
 { Version 6.0. The file system directory acting as a staging area for files
   waiting to be written to CD. A typical path is C:\Documents and Settings\
   username\Local Settings\Application Data\Microsoft\CD Burning. }

 CSIDL_COMMON_ADMINTOOLS = $002f;
 { Version 5.0. The file system directory containing administrative tools
   for all users of the computer. }

 CSIDL_COMMON_ALTSTARTUP = $001e;
 { The file system directory that corresponds to the nonlocalized Startup
   program group for all users. Valid only for Microsoft Windows NT systems. }

 CSIDL_COMMON_APPDATA = $0023;
 { Version 5.0. The file system directory containing application data for
   all users. A typical path is C:\Documents and Settings\All Users\
   Application Data. }

 CSIDL_COMMON_DESKTOPDIRECTORY = $0019;
 { The file system directory that contains files and folders that appear on
   the desktop for all users. A typical path is C:\Documents and Settings\All
   Users\Desktop. Valid only for Windows NT systems. }

 CSIDL_COMMON_DOCUMENTS = $002e;
 { The file system directory that contains documents that are common to all
   users. A typical paths is C:\Documents and Settings\All Users\Documents.
   Valid for Windows NT systems and Microsoft Windows 95 and Windows 98
   systems with Shfolder.dll installed. }

 CSIDL_COMMON_FAVORITES = $001f;
 { The file system directory that serves as a common repository for favorite
   items common to all users. Valid only for Windows NT systems. }

 CSIDL_COMMON_MUSIC = $0035;
 { Version 6.0. The file system directory that serves as a repository for
   music files common to all users. A typical path is C:\Documents and
   Settings\All Users\Documents\My Music. }

 CSIDL_COMMON_PICTURES = $0036;
 { Version 6.0. The file system directory that serves as a repository for
   image files common to all users. A typical path is C:\Documents and
   Settings\All Users\Documents\My Pictures. }

 CSIDL_COMMON_PROGRAMS = $0017;
 { The file system directory that contains the directories for the common
   program groups that appear on the Start menu for all users. A typical
   path is C:\Documents and Settings\All Users\Start Menu\Programs. Valid
   only for Windows NT systems. }

 CSIDL_COMMON_STARTMENU = $0016;
 { The file system directory that contains the programs and folders that
   appear on the Start menu for all users. A typical path is C:\Documents
   and Settings\All Users\Start Menu. Valid only for Windows NT systems. }

 CSIDL_COMMON_STARTUP = $0018;
 { The file system directory that contains the programs that appear in the
   Startup folder for all users. A typical path is C:\Documents and Settings\
   All Users\Start Menu\Programs\Startup. Valid only for Windows NT systems. }

 CSIDL_COMMON_TEMPLATES = $002d;
 { The file system directory that contains the templates that are available
   to all users. A typical path is C:\Documents and Settings\All Users\
   Templates. Valid only for Windows NT systems. }

 CSIDL_COMMON_VIDEO = $0037;
 { Version 6.0. The file system directory that serves as a repository for
   video files common to all users. A typical path is C:\Documents and
   Settings\All Users\Documents\My Videos. }

 CSIDL_COMPUTERSNEARME = $003d;
 { The folder representing other machines in your workgroup. }

 CSIDL_CONNECTIONS = $0031;
 { The virtual folder representing Network Connections, containing network
   and dial-up connections. }

 CSIDL_CONTROLS = $0003;
 { The virtual folder containing icons for the Control Panel applications. }

 CSIDL_COOKIES = $0021;
 { The file system directory that serves as a common repository for
   Internet cookies. A typical path is C:\Documents and Settings\
   username\Cookies. }

 CSIDL_DESKTOP = $0000;
 { The virtual folder representing the Windows desktop, the root
   of the namespace. }

 CSIDL_DESKTOPDIRECTORY = $0010;
 { The file system directory used to physically store file objects on the
   desktop (not to be confused with the desktop folder itself). A typical
   path is C:\Documents and Settings\username\Desktop. }

 CSIDL_DRIVES = $0011;
 { The virtual folder representing My Computer, containing everything on
   the local computer: storage devices, printers, and Control Panel. The
   folder may also contain mapped network drives. }

 CSIDL_FAVORITES = $0006;
 { The file system directory that serves as a common repository for the
   user's favorite items. A typical path is C:\Documents and Settings\
   username\Favorites. }

 CSIDL_FLAG_DONT_UNEXPAND = $2000;
 { Combine with another CSIDL constant to ensure expanding of
   environment variables. }

 CSIDL_FLAG_DONT_VERIFY = $4000;
 { Combine with another CSIDL constant, except for  CSIDL_FLAG_CREATE,
   to return an unverified folder pathwith no attempt to create or
   initialize the folder. }

 CSIDL_FONTS = $0014;
 { A virtual folder containing fonts. A typical path is C:\Windows\Fonts. }

 CSIDL_HISTORY = $0022;
 { The file system directory that serves as a common repository for Internet
  history items. }

 CSIDL_INTERNET = $0001;
 { A viritual folder for Internet Explorer (icon on desktop). }

 CSIDL_INTERNET_CACHE = $0020;
 { Version 4.72. The file system directory that serves as a common
   repository for temporary Internet files. A typical path is C:\Documents
   and Settings\username\Local Settings\Temporary Internet Files. }

 CSIDL_LOCAL_APPDATA = $001c;
 { Version 5.0. The file system directory that serves as a data repository
   for local (nonroaming) applications. A typical path is C:\Documents and
   Settings\username\Local Settings\Application Data. }

 CSIDL_MYDOCUMENTS = $000c;
 { Version 6.0. The virtual folder representing the My
   Documents desktop item. }

 CSIDL_MYMUSIC = $000d;
 { The file system directory that serves as a common repository for
   music files. A typical path is C:\Documents and Settings\User\My
   Documents\My Music. }

 CSIDL_MYPICTURES = $0027;
 { Version 5.0. The file system directory that serves as a common
   repository for image files. A typical path is C:\Documents and
   Settings\username\My Documents\My Pictures. }

 CSIDL_MYVIDEO = $000e;
 { Version 6.0. The file system directory that serves as a common
   repository for video files. A typical path is C:\Documents and
   Settings\username\My Documents\My Videos. }

 CSIDL_NETHOOD = $0013;
 { A file system directory containing the link objects that may exist
   in the My Network Places virtual folder. It is not the same as
   CSIDL_NETWORK, which represents the network namespace root. A typical
   path is C:\Documents and Settings\username\NetHood. }

 CSIDL_NETWORK = $0012;
 { A virtual folder representing Network Neighborhood, the root of the
   network namespace hierarchy. }

 CSIDL_PERSONAL = $0005;
 { Version 6.0. The virtual folder representing the My Documents desktop
   item. This is equivalent to  CSIDL_MYDOCUMENTS.
   Previous to Version 6.0. The file system directory used to physically
   store a user's common repository of documents. A typical path is
   C:\Documents and Settings\username\My Documents. This should be
   distinguished from the virtual My Documents folder in the namespace.
   To access that virtual folder, use SHGetFolderLocation, which returns
   the ITEMIDLIST for the virtual location, or refer to the technique
   described in Managing the File System. }


 CSIDL_PHOTOALBUMS = $0045;
 { Windows Vista. The virtual folder used to store photo albums,
   typically username\My Pictures\Photo Albums. }

 CSIDL_PLAYLISTS = $003f;
 { Windows Vista. The virtual folder used to store play albums,
   typically username\My Music\Playlists. }

 CSIDL_PRINTERS = $0004;
 { The virtual folder containing installed printers. }

 CSIDL_PRINTHOOD = $001b;
 { The file system directory that contains the link objects that can
   exist in the Printers virtual folder. A typical path is C:\Documents
   and Settings\username\PrintHood. }

 CSIDL_PROFILE = $0028;
 { Version 5.0. The user's profile folder. A typical path is C:\Documents
   and Settings\username. Applications should not create files or folders
   at this level; they should put their data under the locations referred
   to by  CSIDL_APPDATA or  CSIDL_LOCAL_APPDATA. }

 CSIDL_PROGRAM_FILES = $0026;
 { Version 5.0. The Program Files folder. A typical path is
   C:\Program Files. }

 CSIDL_PROGRAM_FILES_COMMON = $002b;
 { Version 5.0. A folder for components that are shared across
   applications. A typical path is C:\Program Files\Common. Valid
   only for Windows NT, Windows 2000, and Windows XP systems. Not
   valid for Windows Millennium Edition (Windows Me). }

 CSIDL_PROGRAMS = $0002;
 { The file system directory that contains the user's program groups
   (which are themselves file system directories). A typical path is
   C:\Documents and Settings\username\Start Menu\Programs. }

 CSIDL_RECENT = $0008;
 { The file system directory that contains shortcuts to the user's most
   recently used documents. A typical path is C:\Documents and Settings\
   username\My Recent Documents. To create a shortcut in this folder, use
   SHAddToRecentDocs. In addition to creating the shortcut, this function
   updates the Shell's list of recent documents and adds the shortcut to
   the My Recent Documents submenu of the Start menu. }

 CSIDL_RESOURCES = $0038;
 { Windows Vista. The file system directory that contains resource data.
   A typical path is C:\Windows\Resources. }

 CSIDL_SAMPLE_MUSIC = $0040;
 { Windows Vista. The file system directory that contains sample music.
   A typical path is C:\Documents and Settings\username\My Documents\
   My Music\Sample Music. }

 CSIDL_SAMPLE_PLAYLISTS = $0041;
 { Windows Vista. The file system directory that contains sample playlists.
   A typical path is C:\Documents and Settings\username\My Documents\My
   Music\Sample Playlists. }

 CSIDL_SAMPLE_PICTURES = $0042;
 { Windows Vista. The file system directory that contains sample pictures.
   A typical path is C:\Documents and Settings\username\My Documents\My
   Pictures\Sample Pictures. }

 CSIDL_SAMPLE_VIDEOS = $0043;
 { Windows Vista. The file system directory that contains sample videos.
   A typical path is C:\Documents and Settings\username\My Documents\My
   Videos\Sample Videos. }

 CSIDL_SENDTO = $0009;
 { The file system directory that contains Send To menu items. A typical
   path is C:\Documents and Settings\username\SendTo. }

 CSIDL_STARTMENU = $000b;
 { The file system directory containing Start menu items. A typical path
   is C:\Documents and Settings\username\Start Menu. }

 CSIDL_STARTUP = $0007;
 { The file system directory that corresponds to the user's Startup
   program group. The system starts these programs whenever any user
   logs onto Windows NT or starts Windows 95. A typical path is
   C:\Documents and Settings\username\Start Menu\Programs\Startup. }

 CSIDL_SYSTEM = $0025;
 { Version 5.0. The Windows System folder. A typical path is
   C:\Windows\System32. }

 CSIDL_TEMPLATES = $0015;
 { The file system directory that serves as a common repository for
   document templates. A typical path is C:\Documents and
   Settings\username\Templates. }

 CSIDL_WINDOWS = $0024;
 { Version 5.0. The Windows directory or SYSROOT. This corresponds to
   the %windir% or %SYSTEMROOT% environment variables. A typical
   path is C:\Windows. }

function SHGetSpecialFolderPath; external
 'Shell32.dll' Name 'SHGetSpecialFolderPathA';

{ TCCDSHFolders }

constructor TCCDSHFolders.Create(aOwner: TComponent);
begin
  inherited Create(aOwner);
  FSHFolderPath := EmptyStr;
  FSHFolderID := fiAltstartup;
  SetSHFolderID(fiAdmintools);
end;

function TCCDSHFolders.GetSHFolder(shFolderID: TSHFolderID): string;
begin
  SetSHFolderID(shFolderID);
  result := FSHFolderPath;
end;

procedure TCCDSHFolders.SetSHFolderID(value: TSHFolderID);
var
  buffer: array[0..MAX_PATH] of char;
begin
  if (FSHFolderID <> value) then begin
    FSHFolderID := value;
    if SHGetSpecialFolderPath(GetActiveWindow(),
     buffer,SHFolderIdToCSIDL(value),false) = TRUE then begin
       FSHFolderPath := IncludeTrailingPathDelimiter(buffer);
    end else begin
      FSHFolderPath := EmptyStr;
    end;
  end;
end;

function TCCDSHFolders.SHFolderIdToCSIDL(shFolderID: TSHFolderID): integer;
begin
  case shFolderID of
    fiFonts: result := CSIDL_FONTS;
    fiDrives: result := CSIDL_DRIVES;
    fiSendto: result := CSIDL_SENDTO;
    fiRecent: result := CSIDL_RECENT;
    fiSystem: result := CSIDL_SYSTEM;
    fiHistory: result := CSIDL_HISTORY;
    fiStartup: result := CSIDL_STARTUP;
    fiWindows: result := CSIDL_WINDOWS;
    fiMymusic: result := CSIDL_MYMUSIC;
    fiMyvideo: result := CSIDL_MYVIDEO;
    fiNethood: result := CSIDL_NETHOOD;
    fiProfile: result := CSIDL_PROFILE;
    fiNetwork: result := CSIDL_NETWORK;
    fiCookies: result := CSIDL_COOKIES;
    fiDesktop: result := CSIDL_DESKTOP;
    fiAppdata: result := CSIDL_APPDATA;
    fiControls: result := CSIDL_CONTROLS;
    fiPrinters: result := CSIDL_PRINTERS;
    fiInternet: result := CSIDL_INTERNET;
    fiPrograms: result := CSIDL_PROGRAMS;
    fiPersonal: result := CSIDL_PERSONAL;
    fiStartmenu: result := CSIDL_STARTMENU;
    fiTemplates: result := CSIDL_TEMPLATES;
    fiFavorites: result := CSIDL_FAVORITES;
    fiPlaylists: result := CSIDL_PLAYLISTS;
    fiResources: result := CSIDL_RESOURCES;
    fiPrinthood: result := CSIDL_PRINTHOOD;
    fiBitbucket: result := CSIDL_BITBUCKET;
    fiAdmintools: result := CSIDL_ADMINTOOLS;
    fiAltstartup: result := CSIDL_ALTSTARTUP;
    fiMypictures: result := CSIDL_MYPICTURES;
    fiCdburnArea: result := CSIDL_CDBURN_AREA;
    fiMydocuments: result := CSIDL_MYDOCUMENTS;
    fiPhotoalbums: result := CSIDL_PHOTOALBUMS;
    fiConnections: result := CSIDL_CONNECTIONS;
    fiCommonVideo: result := CSIDL_COMMON_VIDEO;
    fiCommonMusic: result := CSIDL_COMMON_MUSIC;
    fiSampleMusic: result := CSIDL_SAMPLE_MUSIC;
    fiLocalAppdata: result := CSIDL_LOCAL_APPDATA;
    fiProgramFiles: result := CSIDL_PROGRAM_FILES;
    fiSampleVideos: result := CSIDL_SAMPLE_VIDEOS;
    fiCommonAppdata: result := CSIDL_COMMON_APPDATA;
    fiInternetCache: result := CSIDL_INTERNET_CACHE;
    fiCommonStartup: result := CSIDL_COMMON_STARTUP;
    fiCommonPictures: result := CSIDL_COMMON_PICTURES;
    fiSamplePictures: result := CSIDL_SAMPLE_PICTURES;
    fiCommonPrograms: result := CSIDL_COMMON_PROGRAMS;
    fiComputersnearme: result := CSIDL_COMPUTERSNEARME;
    fiCommonDocuments: result := CSIDL_COMMON_DOCUMENTS;
    fiCommonTemplates: result := CSIDL_COMMON_TEMPLATES;
    fiSamplePlaylists: result := CSIDL_SAMPLE_PLAYLISTS;
    fiCommonFavorites: result := CSIDL_COMMON_FAVORITES;
    fiCommonStartmenu: result := CSIDL_COMMON_STARTMENU;
    fiDesktopdirectory: result := CSIDL_DESKTOPDIRECTORY;
    fiCommonAdmintools: result := CSIDL_COMMON_ADMINTOOLS;
    fiCommonAltstartup: result := CSIDL_COMMON_ALTSTARTUP;
    fiProgramFilesCommon: result := CSIDL_PROGRAM_FILES_COMMON;
    fiCommonDesktopdirectory: result := CSIDL_COMMON_DESKTOPDIRECTORY;
  else
    result := -1;
  end;
end;

end.
