unit UCCDMultiAlarms;

interface

uses
  Classes, ExtCtrls, UCCDBaseComp;

type
  TCCDAlarm = class;
  TCCDAlarms = class;
  TRunAlarm = procedure(sender: TObject; alarm: TCCDAlarm) of object;

  TCCDMultiAlarms = class(TCCDBaseComp)
  private
    FTimer: TTimer;
    FActive: boolean;
    FAlarms: TCCDAlarms;
    FOnAlarm: TRunAlarm;
    procedure SetActive(value: boolean);
    procedure CheckAlarms(sender: TObject);
    procedure SetAlarms(alarms: TCCDAlarms);
    function CheckAlarm(alarm: TCCDAlarm): boolean;
    function AlarmInDateRange(alarm: TCCDAlarm): boolean;
  public
    constructor Create(aOwner: TComponent); override;
    destructor Destroy(); override;
  published
    property Active: boolean read FActive write SetActive;
    property Alarms: TCCDAlarms read FAlarms write SetAlarms;
    property OnAlarm: TRunAlarm read FOnAlarm write FOnAlarm;
  end;

  TCCDAlarm = class(TCollectionItem)
  private
    FName,FDescription: string;
    FYear,FMonth,FDay,FHour,FMinute,FSecond: word;
  protected
    function GetDisplayName(): string; override;
  public
    constructor Create(collection: TCollection); override;
  published
    property Day: word read FDay write FDay;
    property Hour: word read FHour write FHour;
    property Year: word read FYear write FYear;
    property Name: string read FName write FName;
    property Month: word read FMonth write FMonth;
    property Minute: word read FMinute write FMinute;
    property Second: word read FSecond write FSecond;
    property Description: string read FDescription write FDescription;
  end;

  TCCDAlarms = class(TCollection)
  private
    FManager: TCCDMultiAlarms;
  protected
    function GetOwner(): TPersistent; override;
  public
    constructor Create(manager: TCCDMultiAlarms);
  end;

implementation

uses
  SysUtils;

{ TCCDMultiAlarms }

constructor TCCDMultiAlarms.Create(aOwner: TComponent);
begin
  inherited Create(aOwner);
  FActive := false;
  FAlarms := TCCDAlarms.Create(self);
  FTimer := TTimer.Create(self);
  with FTimer do begin
    Interval := 1000;
    Enabled := false;
    OnTimer := CheckAlarms;
  end;
end;

destructor TCCDMultiAlarms.Destroy();
begin
  FTimer.Free();
  FAlarms.Free();
  inherited Destroy();
end;

procedure TCCDMultiAlarms.SetActive(value: boolean);
begin
  if (FActive <> value) then begin
    FActive := value;
    if not (csDesigning in Owner.ComponentState) then
      FTimer.Enabled := value;
  end;
end;

procedure TCCDMultiAlarms.SetAlarms(alarms: TCCDAlarms);
begin
  FAlarms.Assign(alarms);
end;

function TCCDMultiAlarms.AlarmInDateRange(alarm: TCCDAlarm): boolean;
var
  year, month, day: word;
begin
  DecodeDate(Date,year,month,day);
  result := (alarm.Year >= year) and
             (alarm.Month >= month);
end;

function TCCDMultiAlarms.CheckAlarm(alarm: TCCDAlarm): boolean;
var
  year, month, day, hour,
  minute, second, msecond: word;
begin
  DecodeDate(Date, year, month, day);
  DecodeTime(Time, hour, minute, second, msecond);
  result := (alarm.Year = year) and (alarm.Month = month)
   and (alarm.Day = day) and (alarm.Hour = hour) and
    (alarm.Minute = minute) and (alarm.Second = second);
end;

procedure TCCDMultiAlarms.CheckAlarms(sender: TObject);
var
  i: integer;
begin
  if not (csDesigning in Owner.ComponentState) then
  begin
    for i := 0 to FAlarms.Count-1 do begin
      if AlarmInDateRange(TCCDAlarm(FAlarms.Items[i])) then
      begin
        if CheckAlarm(TCCDAlarm(FAlarms.Items[i]))
         and Assigned(FOnAlarm) then begin
           FOnAlarm(self,TCCDAlarm(FAlarms.Items[i]));
         end;
      end;
    end;
  end;
end;

{ TCCDAlarms }

constructor TCCDAlarms.Create(manager: TCCDMultiAlarms);
begin
  inherited Create(TCCDAlarm);
  FManager := manager;
end;

function TCCDAlarms.GetOwner(): TPersistent;
begin
  result := FManager;
end;

{ TCCDAlarm }

constructor TCCDAlarm.Create(collection: TCollection);
var
  FMSecond: word;
begin
  inherited Create(collection);
  FName := EmptyStr;
  FDescription := EmptyStr;
  DecodeDate(Date, FYear, FMonth, FDay);
  DecodeTime(Time, FHour, FMinute, FSecond, FMSecond);
end;

function TCCDAlarm.GetDisplayName(): string;
begin
  if (Trim(FName) <> EmptyStr) then
    result := FName
  else
    result := inherited GetDisplayName();
end;

end.
