unit UCCDLabelLink;

interface

uses
  Classes, Messages, Controls, StdCtrls, Graphics,
  UCCDBaseComp, UCCDTypes, UCCDEvents;

type
  TURLType = (utFtp, utHttp, utHttps, utMail);

type
  TCCDLabelLink = class(TLabel)
  private
    FCopy: TColor;
    FActive: TColor;
    FVisited: TColor;
    FErrorMsg: string;
    FUrlType: TURLType;
    FFVisited: boolean;
    FAboutCCD: TAboutCCD;
    FOnError: TCCDOnError;
    function UrlTypeToString(urlType: TURLType): string;
    function OpenUrl(urlType: TURLType; url: string): boolean;
    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
  protected
    procedure Click(); override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property AboutCCD: TAboutCCD read FAboutCCD stored false;
    property OnError: TCCDOnError read FOnError write FOnError;
    property Active: TColor read FActive write FActive default clRed;
    property UrlType: TURLType read FUrlType write FUrlType default utHttp;
    property Visited: TColor read FVisited write FVisited default clPurple;
  end;

implementation

uses
  Windows, ShellApi, SysUtils, UCCDResources;

constructor TCCDLabelLink.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FActive := clRed;
  FUrlType := utHttp;
  FFVisited := false;
  Font.Color := clBlue;
  FVisited := clPurple;
  Cursor := crHandPoint;
  Caption := rsCCDUrlExample;
  Font.Style := Font.Style + [fsUnderline];
end;

procedure TCCDLabelLink.CMMouseEnter(var Message: TMessage);
begin
  if not (csDesigning in ComponentState) then begin
    FCopy := Font.Color;
    Font.Color := FActive;
  end;
end;

procedure TCCDLabelLink.CMMouseLeave(var Message: TMessage);
begin
  if not FFVisited then
    Font.Color := FCopy
  else
    Font.Color := FVisited;
  Font.Style := Font.Style + [fsUnderline];
end;

procedure TCCDLabelLink.Click();
begin
  if not OpenUrl(FUrlType, self.Caption) then begin
    if Assigned(FOnError) then
      FOnError(Self, FErrorMsg);
  end;
  FFVisited := true;
end;

function TCCDLabelLink.OpenUrl(urlType: TURLType; url: string): boolean;
begin
  result := (ShellExecute(GetActiveWindow(), PChar('open'),
   PChar(UrlTypeToString(urlType) + url), nil, nil, SW_MAXIMIZE) <= 32);
  if not result then FErrorMsg := SysErrorMessage(GetLastError());
end;

function TCCDLabelLink.UrlTypeToString(urlType: TURLType): string;
begin
  case urlType of
    utFtp: result := rsCCDUrlTypeFTP;
    utHttp: result := rsCCDUrlTypeHTTP;
    utMail: result := rsCCDUrlTypeMAIL;
    utHttps: result := rsCCDUrlTypeHTTPS;
  end;
end;

end.

