unit UCCDHashes;

interface

uses
  Classes, UCCDBaseComp,
  UCCDTypes, UCCDEvents;

const
  DEFAULT_KBBUFFER = 1024;

type
  THashAlgorithm = (haMD5, haSHA1);

type
  TCCDHashes = class(TCCDBaseComp)
  private
    FAbort: boolean;
    FWorking: boolean;
    FKBBuffer: integer;
    FOnWork: TCCDOnWork;
    FOnAbort: TNotifyEvent;
    FOnWorkEnd: TNotifyEvent;
    FOnWorkBegin: TCCDOnWorkBegin;
  private
    procedure SetKBBuffer(value: integer);
  public
    procedure Abort();
    constructor Create(AOwner: TComponent); override;
    function CalcStringHash(str: string; algorithm: THashAlgorithm): string;
    function CalcFileHash(archivo: string; algorithm: THashAlgorithm): string;
    function CalcStreamHash(stream: TStream; algorithm: THashAlgorithm): string;
  public
    property Working: boolean read FWorking;
  published
    property OnWork: TCCDOnWork read FOnWork write FOnWork;
    property OnAbort: TNotifyEvent read FOnAbort write FOnAbort;
    property OnWorkEnd: TNotifyEvent read FOnWorkEnd write FOnWorkEnd;
    property OnWorkBegin: TCCDOnWorkBegin read FOnWorkBegin write FOnWorkBegin;
    property KBBuffer: integer read FKBBuffer write SetKbBuffer default DEFAULT_KBBUFFER;
  end;

implementation

uses
  Windows, SysUtils, UCCDResources, UCCDExceptions;

type
  ALG_ID = ULONG;
  HCRYPTKEY = ULONG;
  HCRYPTHASH = ULONG;
  HCRYPTPROV = ULONG;
  LPAWSTR = PAnsiChar;
  PHCRYPTKEY = ^HCRYPTKEY;
  PHCRYPTPROV = ^HCRYPTPROV;
  PHCRYPTHASH = ^HCRYPTHASH;

const
  PROV_RSA_FULL = 1;
  HP_HASHVAL = $0002;
  CALG_MD5 = $00008003;
  CALG_SHA1  = $00008004;
  CRYPT_NEWKEYSET = $00000008;

function CryptAcquireContext(
    phProv: PHCRYPTPROV;
    pszContainer: LPAWSTR;
    pszProvider: LPAWSTR;
    dwProvType: DWORD;
    dwFlags: DWORD
  ): BOOL; stdcall;
  external ADVAPI32 name 'CryptAcquireContextA';

function CryptCreateHash(
    hProv: HCRYPTPROV;
    Algid: ALG_ID;
    hKey: HCRYPTKEY;
    dwFlags: DWORD;
    phHash: PHCRYPTHASH
  ): BOOL; stdcall;
  external ADVAPI32 name 'CryptCreateHash';

function CryptHashData(
    hHash: HCRYPTHASH;
    const pbData: PBYTE;
    dwDataLen: DWORD;
    dwFlags: DWORD
  ): BOOL; stdcall;
  external ADVAPI32 name 'CryptHashData';

function CryptGetHashParam(
    hHash: HCRYPTHASH;
    dwParam: DWORD;
    pbData: PBYTE;
    pdwDataLen: PDWORD;
    dwFlags: DWORD
  ): BOOL; stdcall;
  external ADVAPI32 name 'CryptGetHashParam';

function CryptDestroyHash(
    hHash: HCRYPTHASH
  ): BOOL; stdcall;
  external ADVAPI32 name 'CryptDestroyHash';

function CryptReleaseContext(
    hProv: HCRYPTPROV;
    dwFlags: DWORD
  ): BOOL; stdcall;
  external ADVAPI32 name 'CryptReleaseContext';

{ TCCDHashes }

constructor TCCDHashes.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FAbort := false;
  FWorking := false;
  FKBBuffer := DEFAULT_KBBUFFER;
end;

procedure TCCDHashes.Abort();
begin
  FAbort := true;
  FWorking := false;
  if Assigned(FOnAbort)
   then FOnAbort(self);
end;

procedure TCCDHashes.SetKBBuffer(value: integer);
begin
  if (value <> FKBBuffer) then
  begin
    if (value >= 0) then
      FKBBuffer := value
    else
      FKBBuffer := DEFAULT_KBBUFFER;
  end;
end;

function TCCDHashes.CalcStringHash(str: string;
 algorithm: THashAlgorithm): string;
var
  stream: TStringStream;
begin
  result := EmptyStr;
  stream := TStringStream.Create(str);
  try
    result := CalcStreamHash(stream,algorithm);
  finally
    stream.Free;
  end;
end;

function TCCDHashes.CalcFileHash(archivo: string;
 algorithm: THashAlgorithm): string;
var
  stream: TFileStream;
begin
  result:= EmptyStr;
  if FileExists(archivo) then begin
    stream:= TFileStream.Create(archivo,
     fmOpenRead or fmShareDenyWrite);
    try
      result:= CalcStreamHash(stream,algorithm);
    finally
      stream.Free;
    end;
  end else begin
    raise ECCDNoFileException.CreateFmt(
     rsCCDENoFileException, [archivo]);
  end;
end;

function TCCDHashes.CalcStreamHash(stream: TStream;
 algorithm: THashAlgorithm): string;
var
  i: integer;
  buffer: PByte;
  success: BOOL;
  algid: ALG_ID;
  dataLen: DWORD;
  bytesRead: DWORD;
  hProv: HCRYPTPROV;
  hHash: HCRYPTHASH;
  position: longint;
  data: array[1..20] of Byte;
begin
  position := 0;
  FAbort := false;
  FWorking := true;
  result:= EmptyStr;
  success := CryptAcquireContext(@hProv, nil, nil, PROV_RSA_FULL, 0);
  if (not success) then begin
    if GetLastError() = DWORD(NTE_BAD_KEYSET) then
      success := CryptAcquireContext(@hProv, nil, nil, PROV_RSA_FULL,
        CRYPT_NEWKEYSET);
  end;
  if success then begin
    if (algorithm = haMD5) then
    begin
      algid := CALG_MD5;
      dataLen := 16
    end else begin
      algid := CALG_SHA1;
      dataLen := 20;
    end;
    if CryptCreateHash(hProv, algid, 0, 0, @hHash) then
    begin
      if Assigned(FOnWorkBegin) then
        FOnWorkBegin(self, stream.Size);
      GetMem(buffer, FKBBuffer * 1024);
      try
        while not FAbort do begin
          bytesRead:= stream.Read(buffer^, FKBBuffer * 1024);
          Inc(position, bytesRead);
          if (bytesRead = 0) then begin
            if (CryptGetHashParam(hHash, HP_HASHVAL, @data, @dataLen, 0)) then
            begin
              for i := 1 to dataLen do begin
                result := result + LowerCase(IntToHex(Integer(data[i]), 2));
              end;
              break;
            end;
          end;
          if (not CryptHashData(hHash, buffer, bytesRead, 0)) then
            break;
          if Assigned(FOnWork) then
            FOnWork(self,position);
        end;
      finally
        FreeMem(buffer);
      end;
      CryptDestroyHash(hHash);
    end;
    CryptReleaseContext(hProv, 0);
  end;
  FWorking := false;
  if not FAbort and Assigned(FOnWorkEnd) then
    FOnWorkEnd(self);
end;

end.
