{******************************************************************************}
{  DBNavigator 1.0                                                        }
{  Autor:  Luis Bataller - Club Delphi                              }
{  Pgina de descargas: http://www.clubdelphi.com/componentes/                 }
{  Pas de orgen: Espaa, agosto de 2007.                                 }
{                                                                              }
{  TCCDDBNavigator                                                           }
{                                                                              }
{  Componente de acceso a base de datos para la navegacin tipo DBNavigator.  }
{  Adems de los clsicos siguiente, anterior, insertar, eliminar, etc,       }
{  incluye un botn para bsqueda, otro para bsqueda avanzada personalizada y }
{  una gestin de bookmarks. Tambin tiene una barra de estado con informacin }
{  de la posicin actual del registro.                                         }
{******************************************************************************}
unit UCCDDBNavigator;

interface

uses
  SysUtils, Classes, Controls, ToolWin, ComCtrls, db,
  ExtCtrls, windows, Menus, stdCtrls, messages, forms,
  buttons, Graphics;

const
  // Formato del texto de la barra de estado
  RECORDSTATUSFORMAT = 'Registro %d de %d';
  RECORDSTATUSFORMATNORECNO = '%d Registros';
  RECORDSTATUSINSERT = 'Nuevo registro';
  RECORDSTATUSEDIT = 'Editando registro %d de %d';
  RECORDSTATUSEDITNORECNO = 'Editando, %d Registros';
  RECORDCONFIRMDELETE = 'Est seguro que quiere eliminar el registro?';
  RECORDCONFIRMCAPTION = 'Confirmacin';
  SEARCHCAPTION = 'Buscar por %s';
type
  // Enlace con la base de datos
  TNavDataLink = class;
  // Tipo de control para los botones
  TButtonType = TSpeedButton;
  // Tipo de control para la barra de estado
  TStatusBarType = TStatusBar;
  // Tipo de dato del control de bsqueda de fechas
  TDateEditSearchType = TDateTimePicker;
  // Tipos de botones disponibles
  TButtonsTypes = (lbnSearch, lbnAdvanceSearch, lbnAuto, lbnFirst, lbnPrevious, lbnNext, lbnLast,
                   lbnAdd, lbnDelete, lbnEdit, lbnPost, lbnCancel,
                   lbnAddBookmark, lbnGotoBookmark, lbnDeleteBookmark, lbnRefresh);
  // Botones Visibles
  TVisibleButtons = set of TButtonsTypes;
  // Ttulos de los botonoes
  TButtonsNames = TStringList;
  // Referencias a los botones
  AButtons = array[TButtonsTypes] of TButtonType;
  // Clase Principal
  TCCDDBNavigator = class(TPanel)
  private
    FDataLink : TNavDataLink;
    FCreate : Boolean;
    FVisibleButtons: TVisibleButtons;
    FCaptions: TButtonsNames;
    FButtonNames : TButtonsNames;
    FButtons: AButtons;
    FImages: TImageList;
    FShowCaptions: Boolean;
    FDataSource: TDataSource;
    FCaptionsAsHints: Boolean;
    FOnClickSearch: TNotifyEvent;
    FOnClickAuto: TNotifyEvent;
    FDisabledImages: TImageList;
    FConfirmDelete: Boolean;
    FStatusBar: Boolean;
    vStatusBar: TStatusBarType;
    PanelHint : TStatusPanel;
    PanelRecord : TStatusPanel;
    FBookMarkDisplayFields: string;
    fBookMarks : TList;
    MenuBookmark : TPopupMenu;
    MenuBookmarkDelete : TPopupMenu;
    MenuSearch : TPopupMenu;
    FieldSearchNo : Integer;
    FEditSearch : TEdit;
    FDateEditSearch : TDateEditSearchType;
    FOwnerMouseMove : TMouseMoveEvent;
    ControlLoaded : Boolean;
    ImagesLoaded : Boolean;
    FEnabled: Boolean;
    procedure SetVisibleButtons(const Value: TVisibleButtons);
    procedure SetCaptions(const Value: TButtonsNames);
    procedure AddNames;
    procedure CreateButton(idButton : TButtonsTypes);
    procedure RemoveButton(idButton : TButtonsTypes);
    procedure SetImages(const Value: TImageList);
    procedure SetShowCaptions(const Value: Boolean);
    procedure SetDataSource(const Value: TDataSource);
    procedure SetCaptionsAsHints(const Value: Boolean);
    procedure SetOnClickAuto(const Value: TNotifyEvent);
    procedure SetOnClickSearch(const Value: TNotifyEvent);
    procedure SetDisabledImages(const Value: TImageList);
    procedure SetConfirmDelete(const Value: Boolean);
    procedure SetStatusBar(const Value: Boolean);
    procedure CreateStatusBar;
    procedure SetBookMarkDisplayFields(const Value: string);
    procedure CreateMenuSearch;
    procedure DoMenuSearch(Sender : TObject);
    procedure DoOnExitSearch(Sender : TObject);
    procedure DoOnKeyPressSearch(Sender : TObject; var Key : char);
    procedure DoOnChangeSearch(Sender : TObject);
    { Private declarations }
  protected
    { Protected declarations }
    procedure CreateButtons;
    procedure RefreshButtons;
    procedure SetParent(AParent: TWinControl); override;
    procedure Resize; override;
    procedure DataChanged;
    procedure EditingChanged;
    procedure ActiveChanged;
    procedure ButtonEvent(Sender : TObject);
    procedure ButtonEnter(Sender: TObject; Shift: TShiftState; X,Y: Integer);
    procedure ButtonLeave(Sender: TObject; Shift: TShiftState; X,Y: Integer);
    procedure ShowMenu(Sender: TObject);
    procedure PrintRecordStatus;
    procedure AddBookmark;
    procedure DoDeleteBookmark(Sender : TObject);
    procedure DoGotoBookmark(Sender : TObject);
    procedure CreateWnd; override;
    function ButtonByName(buttonName : string) : TButtonType;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure Loaded; override;
    procedure LoadImages;
    procedure SetEnabled(Value: Boolean); override;
  public
    { Public declarations }
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
  published
    { Published declarations }
    property VisibleButtons : TVisibleButtons read FVisibleButtons write SetVisibleButtons;
    property Captions : TButtonsNames read FCaptions write SetCaptions;
    property Images : TImageList read FImages write SetImages;
    property DisabledImages : TImageList read FDisabledImages write SetDisabledImages;
    property ShowCaptions : Boolean read FShowCaptions write SetShowCaptions;
    property DataSource : TDataSource read FDataSource write SetDataSource;
    property CaptionsAsHints : Boolean read FCaptionsAsHints write SetCaptionsAsHints;
    property OnClickSearch : TNotifyEvent read FOnClickSearch write SetOnClickSearch;
    property OnClickAuto : TNotifyEvent read FOnClickAuto write SetOnClickAuto;
    property ConfirmDelete: Boolean read FConfirmDelete write SetConfirmDelete;
    property StatusBar: Boolean read FStatusBar write SetStatusBar;
    property BookMarkDisplayFields: string read FBookMarkDisplayFields write SetBookMarkDisplayFields;
    property Enabled: Boolean read FEnabled write SetEnabled;
  end;

TNavDataLink = class(TDataLink)
  private
    FNavigator: TCCDDBNavigator;
  protected
    procedure EditingChanged; override;
    procedure DataSetChanged; override;
    procedure ActiveChanged; override;
  public
    constructor Create(Nav: TCCDDBNavigator);
    destructor Destroy; override;
  end;


implementation


{ TCCDDBNavigator }

// Nombres y captions por defecto de los botones
procedure TCCDDBNavigator.AddNames;
begin
  FButtonNames.Clear;
  FButtonNames.Add('Search');
  FButtonNames.Add('AdvanceSearch');
  FButtonNames.Add('Auto');
  FButtonNames.Add('First');
  FButtonNames.Add('Previous');
  FButtonNames.Add('Next');
  FButtonNames.Add('Last');
  FButtonNames.Add('Add');
  FButtonNames.Add('Delete');
  FButtonNames.Add('Edit');
  FButtonNames.Add('Post');
  FButtonNames.Add('Cancel');
  FButtonNames.Add('AddBookMark');
  FButtonNames.Add('GotoBookMark');
  FButtonNames.Add('DeleteBookMark');
  FButtonNames.Add('Refresh');

  FCaptions.Clear;
  FCaptions.Add('Buscar');
  FCaptions.Add('Bsqueda Avanzada');
  FCaptions.Add('Autocdigo');
  FCaptions.Add('Primero');
  FCaptions.Add('Anterior');
  FCaptions.Add('Siguiente');
  FCaptions.Add('ltimo');
  FCaptions.Add('Aadir');
  FCaptions.Add('Eliminar');
  FCaptions.Add('Editar');
  FCaptions.Add('Guardar');
  FCaptions.Add('Cancelar');
  FCaptions.Add('Aadir Marcador');
  FCaptions.Add('Ir a Marcador');
  FCaptions.Add('Eliminar Marcador');
  FCaptions.Add('Actualizar');
end;

// Devuelve la referencia a un botn buscando por su nombre
function TCCDDBNavigator.ButtonByName(buttonName: string): TButtonType;
var
  button : TComponent;
begin
  button := FindComponent(Name + buttonName);
  if (button <> nil) then
    Result := button as TButtonType
  else
    Result := nil;
end;

// Constructor
constructor TCCDDBNavigator.Create(AOwner: TComponent);
var
  i : TButtonsTypes;
begin
  ControlLoaded := False;
  ImagesLoaded := False;
  FEditSearch := nil;
  FDateEditSearch := nil;
  fBookMarks := TList.Create;
  vStatusBar := nil;
  for i := lbnSearch to lbnRefresh do
    FButtons[i] := nil;

  FButtonNames := TStringList.Create;
  FCaptions := TStringList.Create;
  AddNames;
  inherited;
  FDataLink := TNavDataLink.Create(Self);
  FCreate := False;
  // Botones visibles por defecto
  VisibleButtons := [lbnSearch, lbnAuto, lbnFirst, lbnPrevious, lbnNext, lbnLast,
                   lbnAdd, lbnDelete, lbnEdit, lbnPost, lbnCancel, lbnRefresh];
  Align := alNone;
  ShowCaptions := False;
  BevelOuter := bvNone;
  MenuBookmark := TPopupMenu.Create(Self);
  MenuBookmarkDelete := TPopupMenu.Create(Self);
  MenuSearch := TPopupMenu.Create(Self);
  ConfirmDelete := True;
  Caption := '';
  CaptionsAsHints := True;
  ShowHint := True;
end;


// Funcin que crea un botn del tipo idButton
procedure TCCDDBNavigator.CreateButton(idButton: TButtonsTypes);
begin
  if Assigned(Parent) and (Name <> '')
     and (FButtons[idButton] = nil) then
  begin
    FButtons[idButton] := TButtonType.Create(self);
    if Ord(idButton) < FCaptions.Count then
      FButtons[idButton].Caption := FCaptions[Ord(idButton)];
    FButtons[idButton].Name := Name + FButtonNames[Ord(idButton)];
    FButtons[idButton].OnClick := ButtonEvent;
    FButtons[idButton].OnMouseMove := ButtonEnter;
    FButtons[idButton].Tag := Ord(idButton);
    InsertControl(FButtons[idButton]);
  end;

end;

// Crear todos los botones
procedure TCCDDBNavigator.CreateButtons;
var
  i : TButtonsTypes;
begin
  if not FCreate and Assigned(Parent) and (Name <> '') then
  begin
    for i := lbnSearch to lbnRefresh do
    begin
      CreateButton(i);
      FButtons[i].Enabled := False;
    end;
    FCreate := True;
  end;
end;


// Destructor: Liberamos objetos
destructor TCCDDBNavigator.Destroy;
var
  i : TButtonsTypes;
  j : Integer;
begin
  for j := 0 to fBookMarks.Count - 1 do
     if Assigned(DataSource) and Assigned(DataSource.DataSet) then
       DataSource.DataSet.FreeBookmark(fBookMarks[j]);

  FreeAndNil(FDataLink);
  FreeAndNil(FButtonNames);
  FreeAndNil(FCaptions);
  for i := lbnSearch to lbnRefresh do
    RemoveButton(i);
  if vStatusBar <> nil then
    FreeAndNil(vStatusBar);


  FreeAndNil(fBookMarks);
  FreeAndNil(MenuBookmark);
  FreeAndNil(MenuBookmarkDelete);
  FreeAndNil(MenuSearch);
  if (FEditSearch <> nil) and (Parent <> nil) then
  begin
    Parent.RemoveControl(FEditSearch);
    FreeAndNil(FEditSearch);
  end;
  inherited;
end;


// Redibuja los botones
procedure TCCDDBNavigator.RefreshButtons;
var
  i : TButtonsTypes;
  Position : Integer;
  Visibles : Integer;
  btnWidth : Integer;
  btnHeight : Integer;
  wRest : Integer;
begin
  if ControlLoaded then
  begin
    if Caption <> '' then // eliminamos el caption del panel
      Caption := '';
    Visibles := 0;
    for i := lbnRefresh downto lbnSearch do
    begin
      if (i in VisibleButtons) then
      begin
        CreateButton(i);
        Inc(Visibles); // Para calcular el tamao de los botones
      end
      else if not (i in VisibleButtons) then
        RemoveButton(i);

    end;

    // Crear la barra de estado
    CreateStatusBar;
    if (vStatusBar <> nil) and StatusBar then
      btnHeight := Height - vStatusBar.Height - (BorderWidth * 2)
    else
      btnHeight := Height - (BorderWidth * 2);

    Position := 0;
    if Visibles > 0 then
    begin
      btnWidth := (Width - (BorderWidth * 2)) div Visibles;
      wRest := (Width - (BorderWidth * 2)) mod Visibles;  // para ajustar los pixeles sobrantes
    end
    else begin
      btnWidth := 0;
      wRest := 0;
    end;

    // propiedades de tamao de los botones
    for i := lbnSearch to lbnRefresh do
    begin
      if fButtons[i] <> nil then // Cambiar el resto de propiedades
      begin
        if showCaptions and (Ord(i) < fCaptions.Count) then
          fButtons[i].Caption := fCaptions[Ord(i)]
        else
          fButtons[i].Caption := '';
        fButtons[i].Flat := True;
        fButtons[i].Transparent := True;
        fButtons[i].Top := BorderWidth;
        fButtons[i].Height := btnHeight;
        fButtons[i].Left := Position + BorderWidth;
        if wRest > 0 then
        begin
          fButtons[i].Width := btnWidth + 1;
          Dec(wRest);
        end
        else
          fButtons[i].Width := btnWidth;

        Inc(Position, fButtons[i].Width);

        if (Ord(i) < fCaptions.Count) and FCaptionsAsHints then
          fButtons[i].Hint := fCaptions[Ord(i)];
        fButtons[i].ShowHint := FCaptionsAsHints;
      end;

    end;
    LoadImages;
  end;


end;

// Eliminar un botn
procedure TCCDDBNavigator.RemoveButton(idButton: TButtonsTypes);
begin
  if FButtons[idButton] <> nil then
  begin
    Self.RemoveComponent(fButtons[idButton]);
    FreeAndNil(fButtons[idButton]);
  end;
end;

procedure TCCDDBNavigator.Resize;
begin
  inherited;
  RefreshButtons;
end;


procedure TCCDDBNavigator.SetCaptions(const Value: TButtonsNames);
begin
  FCaptions.Assign(Value);
end;


procedure TCCDDBNavigator.SetDataSource(const Value: TDataSource);
begin
  FDataSource := Value;
  FDataLink.DataSource := Value;
  if not (csLoading in ComponentState) then
    ActiveChanged;

  if Value <> nil then Value.FreeNotification(self);
end;

procedure TCCDDBNavigator.SetImages(const Value: TImageList);
begin
  FImages := Value;
  ImagesLoaded := False;
  RefreshButtons;
end;

procedure TCCDDBNavigator.SetParent(AParent: TWinControl);
begin
  inherited;
  CreateButtons;

end;



procedure TCCDDBNavigator.SetShowCaptions(const Value: Boolean);
begin
  FShowCaptions := Value;
  RefreshButtons;
end;

procedure TCCDDBNavigator.SetVisibleButtons(const Value: TVisibleButtons);
begin
  FVisibleButtons := Value;
  ImagesLoaded := False;
  RefreshButtons;
end;

procedure TCCDDBNavigator.ActiveChanged;
begin
  DataChanged;
end;

// Cambia el estado de los botones a activado o desactivado segn
// el estado del dataset
procedure TCCDDBNavigator.DataChanged;
var
  UpEnable, DnEnable: Boolean;
  CanModify : Boolean;
begin
  if Assigned(FDataLink) and Assigned(FDataLink.DataSet) then
  begin
    PrintRecordStatus;
    UpEnable := Enabled and FDataLink.Active and not FDataLink.DataSet.BOF;
    DnEnable := Enabled and FDataLink.Active and not FDataLink.DataSet.EOF;
    CanModify := Enabled and FDataLink.Active and FDataLink.DataSet.CanModify;

    if FButtons[lbnSearch] <> nil then
      FButtons[lbnSearch].Enabled := UpEnable or DnEnable;
    if FButtons[lbnAdvanceSearch] <> nil then
      FButtons[lbnAdvanceSearch].Enabled := UpEnable or DnEnable;
    if FButtons[lbnAuto] <> nil then
      FButtons[lbnAuto].Enabled := CanModify and not FDataLink.Editing;
    if FButtons[lbnFirst] <> nil then
      FButtons[lbnFirst].Enabled := upEnable;
    if FButtons[lbnPrevious] <> nil then
      FButtons[lbnPrevious].Enabled := upEnable;
    if FButtons[lbnNext] <> nil then
      FButtons[lbnNext].Enabled := DnEnable;
    if FButtons[lbnLast] <> nil then
      FButtons[lbnLast].Enabled := DnEnable;
    if FButtons[lbnAdd] <> nil then
      FButtons[lbnAdd].Enabled := CanModify and not FDataLink.Editing;
    if FButtons[lbnDelete] <> nil then
      FButtons[lbnDelete].Enabled := CanModify;
    if FButtons[lbnEdit] <> nil then
      FButtons[lbnEdit].Enabled := CanModify and not FDataLink.Editing;
    if FButtons[lbnPost] <> nil then
      FButtons[lbnPost].Enabled := CanModify and FDataLink.Editing;
    if FButtons[lbnCancel] <> nil then
      FButtons[lbnCancel].Enabled := CanModify and FDataLink.Editing;
    if FButtons[lbnAddBookMark] <> nil then
      FButtons[lbnAddBookMark].Enabled := Enabled and FDataLink.Active and not FDataLink.Editing;
    if FButtons[lbnGotoBookMark] <> nil then
      FButtons[lbnGotoBookMark].Enabled := Enabled and FDataLink.Active and not FDataLink.Editing;
    if FButtons[lbnDeleteBookMark] <> nil then
      FButtons[lbnDeleteBookMark].Enabled := Enabled and FDataLink.Active and not FDataLink.Editing;
    if FButtons[lbnRefresh] <> nil then
      FButtons[lbnRefresh].Enabled := Enabled and FDataLink.Active;
    CreateMenuSearch;

  end;
end;

procedure TCCDDBNavigator.EditingChanged;
begin
  DataChanged;
end;


// Aadimos un bookmark. En el men se mostrarn los valores de
// de los campos que estn en BookMarkDisplayFields (separados por ';')
procedure TCCDDBNavigator.AddBookMark;
var
  menuCaption : string;
  nFields : TStringList;
  i : Integer;
  menuItem : TMenuItem;
begin
  if Assigned(DataSource) and Assigned(DataSource.DataSet) and
     DataSource.Dataset.Active then
  begin
    nFields := TStringList.Create;
    nFields.Delimiter := ';';
    nFields.DelimitedText := BookMarkDisplayFields;

    with DataSource.DataSet do
    begin
      FBookMarks.Add(GetBookmark);
      menuCaption := IntToStr(FBookMarks.Count);
      for i := 0 to nFields.Count - 1 do
      begin
        if findField(nFields[i]) <> nil then
          menuCaption := menuCaption + '|' + FieldByName(nFields[i]).AsString;
      end;
      nFields.Free;

      MenuItem := TMenuItem.Create(MenuBookmark);
      MenuItem.Caption := menuCaption;
      MenuItem.Tag := FBookMarks.Count - 1; // en el tag guardamos la posicin del bookmark
      MenuItem.OnClick := DoGotoBookmark;
      MenuBookmark.Items.Add(MenuItem);

      MenuItem := TMenuItem.Create(MenuBookmarkDelete);
      MenuItem.Caption := menuCaption;
      MenuItem.Tag := FBookMarks.Count - 1;
      MenuItem.OnClick := DoDeleteBookmark;
      MenuBookmarkDelete.Items.Add(MenuItem);
    end;

  end;

end;

// Eliminamos un bookmark del men
procedure TCCDDBNavigator.DoDeleteBookmark(Sender : TObject);
var
  item : Integer;
  i : Integer;
begin
  if Assigned(DataSource) and Assigned(DataSource.DataSet) and
     DataSource.Dataset.Active then
  begin
    item := (Sender as TMenuItem).Tag;
    if item >= 0 then
    begin
      // restamos 1 al tag de las posiciones mayores del que vamos a eliminar
      for i := 0 to MenuBookMark.Items.Count - 1 do
        if MenuBookMark.items[i].Tag > item then
          MenuBookMark.Items[i].Tag := MenuBookMark.Items[i].Tag - 1;
      for i := 0 to MenuBookMarkDelete.Items.Count - 1 do
        if MenuBookMarkDelete.items[i].Tag > item then
          MenuBookMarkDelete.Items[i].Tag := MenuBookMarkDelete.Items[i].Tag - 1;
      DataSource.DataSet.FreeBookmark(FBookMarks[item]);
      FBookMarks.Delete(Item);
      MenuBookMark.Items.Delete(MenuBookMark.Items.IndexOf(MenuBookMark.Items.Find((Sender as TMenuItem).Caption)));
      MenuBookMarkDelete.Items.Delete(MenuBookMarkDelete.Items.IndexOf(MenuBookMarkDelete.Items.Find((Sender as TMenuItem).Caption)));
    end;
  end;
end;

// Vamos al registro guardado en un bookmark
procedure TCCDDBNavigator.DoGotoBookmark(Sender : TObject);
var
  item : Integer;
begin
  if Assigned(DataSource) and Assigned(DataSource.DataSet) and
     DataSource.Dataset.Active then
  begin
    item := (Sender as TMenuItem).Tag;
    if item >= 0 then
      DataSource.DataSet.GotoBookmark(FBookMarks[item]);
  end;
end;


// Gestin de los botones. Lo que hace cada uno.
procedure TCCDDBNavigator.ButtonEvent(Sender: TObject);

  function AskDelete : boolean;
  begin
    Result := False;
    if ConfirmDelete then
    begin
      if Application.MessageBox(RECORDCONFIRMDELETE,
                       RECORDCONFIRMCAPTION, MB_YESNO) = ID_YES  then
        Result := True;

    end
    else Result := True;
  end;

begin
  if (DataSource <> nil) and (DataSource.State <> dsInactive) and
      Assigned(DataSource.DataSet) then
    with DataSource.DataSet do
    begin
      case TButtonsTypes((Sender as TButtonType).Tag) of
        lbnAdvanceSearch: if Assigned(OnClickSearch) then FOnClickSearch(Sender);
        lbnAuto: if Assigned(OnClickAuto) then FOnClickAuto(Sender);
        lbnFirst: First;
        lbnPrevious: Prior;
        lbnNext: Next;
        lbnLast: Last;
        lbnAdd: Append;
        lbnDelete: if AskDelete then Delete;
        lbnEdit: Edit;
        lbnPost: Post;
        lbnCancel: Cancel;
        lbnAddBookmark: AddBookmark;
        lbnRefresh: Refresh
        else ShowMenu(Sender);
      end;

    end;
end;

procedure TCCDDBNavigator.SetCaptionsAsHints(const Value: Boolean);
begin
  FCaptionsAsHints := Value;
  RefreshButtons;
end;

procedure TCCDDBNavigator.SetOnClickAuto(const Value: TNotifyEvent);
begin
  FOnClickAuto := Value;
end;

procedure TCCDDBNavigator.SetOnClickSearch(const Value: TNotifyEvent);
begin
  FOnClickSearch := Value;
end;

procedure TCCDDBNavigator.SetDisabledImages(const Value: TImageList);
begin
  FDisabledImages := Value;
  ImagesLoaded := False;
  RefreshButtons;
end;

procedure TCCDDBNavigator.SetConfirmDelete(const Value: Boolean);
begin
  FConfirmDelete := Value;
end;

procedure TCCDDBNavigator.SetStatusBar(const Value: Boolean);
begin
  FStatusBar := Value;
  RefreshButtons;
end;

// Creacin de la barra de estado. Slo se muestra si
// se activa en la propiedad StatusBar
procedure TCCDDBNavigator.CreateStatusBar;
begin
  if Assigned(Parent) and (Name <> '')
     and (vStatusBar = nil) and StatusBar then
  begin
    vStatusBar := TStatusBarType.Create(self);
    InsertControl(vStatusBar);
    PanelRecord := vStatusBar.Panels.Add;
    PanelRecord.Width := Width div 2;
    PanelHint := vStatusBar.Panels.Add;
    PanelHint.Alignment := taRightJustify;
  end
  else if (vStatusBar <> nil) and not StatusBar then
  begin
    RemoveControl(vStatusBar);
    FreeAndNil(vStatusBar);
  end;
end;

// Mostrar la informacin en la barra de estado
procedure TCCDDBNavigator.PrintRecordStatus;
var
  RecNo, RecordCount : Integer;
begin
  if FEditSearch = nil then // Si no se est buscando, al buscar se usa el panel como ttulo de la bsqueda
  begin
    if Assigned(DataSource) and Assigned(DataSource.DataSet) and
       DataSource.DataSet.Active and (vStatusBar <> nil) then
    begin
      if DataSource.DataSet.State = dsInsert then
        PanelRecord.Text := RECORDSTATUSINSERT
      else begin
        RecNo := DataSource.DataSet.RecNo;
        RecordCount := DataSource.DataSet.RecordCount;
        if (RecNo > -1) and (DataSource.DataSet.State = dsEdit) then
          PanelRecord.Text := Format(RECORDSTATUSEDIT, [RecNo, RecordCount])
        else if (RecNo = -1) and (DataSource.DataSet.State = dsEdit) then
          PanelRecord.Text := Format(RECORDSTATUSEDITNORECNO, [RecordCount])
        else if (RecNo > -1) and (DataSource.DataSet.State = dsBrowse) then
          PanelRecord.Text := Format(RECORDSTATUSFORMAT, [RecNo, RecordCount])
        else if (RecNo = -1) and (DataSource.DataSet.State = dsBrowse) then
          PanelRecord.Text := Format(RECORDSTATUSFORMATNORECNO, [RecordCount]);
      end;
    end;
  end;


end;

// Para mostrar la informacin sobre el botn en la barra de estado
procedure TCCDDBNavigator.ButtonEnter(Sender: TObject; Shift: TShiftState; X,Y: Integer);
begin
  if CaptionsAsHints and (vStatusBar <> nil) then
    PanelHint.Text := (Sender as TButtonType).Hint;
end;

// Se borra la informacin de la barra de estado cuando se sale del botn
procedure TCCDDBNavigator.ButtonLeave(Sender: TObject; Shift: TShiftState; X,Y: Integer);
begin
  if vStatusBar <> nil then
    PanelHint.Text := '';
  if Assigned(FOwnerMouseMove) then
    FOwnerMouseMove(Sender, Shift, X, Y);
end;

procedure TCCDDBNavigator.SetBookMarkDisplayFields(const Value: string);
begin
  FBookMarkDisplayFields := Value;
end;

// Creacin del men de bsquedas. Se muestran todos los campos con
// el valor de su "DisplayLabel"
procedure TCCDDBNavigator.CreateMenuSearch;
var
  MenuItem : TMenuItem;
  i : Integer;
begin
  if Assigned(DataSource) and Assigned(DataSource.DataSet) then
  begin
    MenuSearch.Items.Clear;
    for i := 0 to DataSource.DataSet.FieldCount - 1 do
      if DataSource.DataSet.Fields[i].Visible then
      begin
        MenuItem := TMenuItem.Create(MenuSearch);
        MenuItem.Caption := DataSource.DataSet.Fields[i].DisplayLabel;
        MenuItem.Tag := i;
        MenuItem.OnClick := DoMenuSearch;
        MenuSearch.Items.Add(MenuItem);
      end;
  end;

end;

// Al escoger un campo para la bsqueda, se muestra el control segn el
// tipo de campo (edit para texto y nmeros y  DateTimePicker para las fechas)
procedure TCCDDBNavigator.DoMenuSearch(Sender : TObject);
  procedure CreateEdit;
  begin
    if FEditSearch = nil then
    begin
      FEditSearch := TEdit.Create(Parent);
      FEditSearch.Name := Name + 'EditSearch';
      FEditSearch.Text := '';
      Parent.InsertControl(FEditSearch);
      FEditSearch.SetFocus;
      FEditSearch.Left := Left;
      FEditSearch.Top := Top + Height;
      FEditSearch.OnExit := DoOnExitSearch;
      FEditSearch.OnKeyPress := DoOnKeyPressSearch;
      FEditSearch.OnChange := DoOnChangeSearch;
    end;
  end;
  procedure CreateDate;
  begin
    if FDateEditSearch = nil then
    begin
      FDateEditSearch := TDateEditSearchType.Create(Parent);
      FDateEditSearch.Name := Name + 'DateEditSearch';
      FDateEditSearch.Date := Date;
      Parent.InsertControl(FDateEditSearch);
      FDateEditSearch.SetFocus;
      FDateEditSearch.Left := Left;
      FDateEditSearch.Top := Top + Height;
      FDateEditSearch.OnExit := DoOnExitSearch;
      FDateEditSearch.OnKeyPress := DoOnKeyPressSearch;
      FDateEditSearch.OnChange := DoOnChangeSearch;
    end;
  end;
begin

  if Assigned(DataSource) and Assigned(DataSource.DataSet) then
  begin
    FieldSearchNo := (Sender as TComponent).Tag;
    if (DataSource.DataSet.Fields[FieldSearchNo].DataType = ftDate) or
       (DataSource.DataSet.Fields[FieldSearchNo].DataType = ftDateTime) then
      CreateDate
    else
      CreateEdit;
    PanelRecord.Text := Format(SEARCHCAPTION, [DataSource.DataSet.Fields[FieldSearchNo].DisplayLabel]);
  end;

end;

// Al salir de la bsqueda
procedure TCCDDBNavigator.DoOnExitSearch(Sender: TObject);
begin
  if FEditSearch <> nil then
  begin
    FEditSearch.OnExit := nil;
    FEditSearch.OnChange := nil;
    FEditSearch.OnKeyPress := nil;
    Parent.RemoveControl(FEditSearch);
    FreeAndNil(FEditSearch);
  end;
  if FDateEditSearch <> nil then
  begin
    FDateEditSearch.OnExit := nil;
    FDateEditSearch.OnChange := nil;
    FDateEditSearch.OnKeyPress := nil;
    Parent.RemoveControl(FDateEditSearch);
    FreeAndNil(FDateEditSearch);
  end;
end;

// Al salir del control de bsqueda (con enter o escape) se elimina el control
procedure TCCDDBNavigator.DoOnKeyPressSearch(Sender: TObject;
  var Key: char);
begin
  if (key = #13) or (key = #27) then
  begin
    try
      if FEditSearch <> nil then
        FEditSearch.OnExit := nil;
      if FDateEditSearch <> nil then
        FDateEditSearch.OnExit := nil;
      Parent.Perform (WM_NEXTDLGCTL, 1, 0);
      DoOnExitSearch(Sender);
    except
    end;
//    FEditSearch.Visible := False;
  end;

end;

procedure TCCDDBNavigator.CreateWnd;
begin
  inherited;
  ImagesLoaded := False;
  ControlLoaded := True;
  RefreshButtons;
  FOwnerMouseMove := OnMouseMove;
  OnMouseMove := ButtonLeave;
  Enabled := True;

end;


// Buscar al introducir texto en el control de bsqueda. En caso de
// que el campo sea de tipo texto se hace una bsqueda por aproximacin

procedure TCCDDBNavigator.DoOnChangeSearch(Sender: TObject);
  function ClearDateText(d : string) : string;
  begin
    while Pos(' ', d) > 0 do
      Delete(d, Pos(' ', d), 1);
    Result := d;
  end;

  procedure SearchLookup(Field : TField; data : string);
  var
    bm : TBookmark;
  begin
    if Assigned(Field.LookupDataSet) then
    begin
      with Field.LookupDataSet do
      begin
        DisableControls;
        bm := getBookMark;
        if Locate(Field.LookupResultField, data, [loCaseInsensitive, loPartialKey]) then
          Field.DataSet.Locate(Field.KeyFields,
                               FieldByName(Field.LookupKeyFields).AsVariant, []);
        GotoBookmark(bm);
        FreeBookmark(bm);
        EnableControls;
      end;
    end;
  end;
var
  vDate : TDate;
begin
  if Assigned(DataSource) and Assigned(DataSource.DataSet) then
  begin
    if FEditSearch <> nil then
    begin
      try
        if DataSource.DataSet.Fields[FieldSearchNo].FieldKind = fkLookup then
          SearchLookup(DataSource.DataSet.Fields[FieldSearchNo], FEditSearch.Text)
        else
          DataSource.DataSet.Locate(DataSource.DataSet.Fields[FieldSearchNo].FieldName,
                                FEditSearch.Text, [loCaseInsensitive, loPartialKey]);
      except
      end;
    end;
    if (FDateEditSearch <> nil) then
    begin
      try
        vDate := StrToDate(ClearDateText(DateToStr(FDateEditSearch.Date)));
        DataSource.DataSet.Locate(DataSource.DataSet.Fields[FieldSearchNo].FieldName,
                             vDate, [loCaseInsensitive, loPartialKey]);
      except
      end;
    end;
  end;

end;

procedure TCCDDBNavigator.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited;
  if Operation = opRemove then
  begin
    if AComponent = FDataSource then
    begin
      FDataSource := nil;
      if FDataLink <> nil then
        FDataLink.DataSource := nil;
    end;
    if AComponent = FDataSource then
      FDataSource := nil;
  end;
end;

// Muestra el men de bookmarks
procedure TCCDDBNavigator.ShowMenu(Sender: TObject);
var
  p : TPoint;
  btn : TButtonType;
begin
  btn := Sender as TButtonType;
  p.X := btn.Left;
  p.Y := btn.Height;
  if btn = FButtons[lbnSearch] then
    MenuSearch.Popup(ClientToScreen(p).X, ClientToScreen(p).Y);
  if btn = FButtons[lbnGotoBookMark] then
    MenuBookmark.Popup(ClientToScreen(p).X, ClientToScreen(p).Y);
  if btn = FButtons[lbnDeleteBookMark] then
    MenuBookmarkDelete.Popup(ClientToScreen(p).X, ClientToScreen(p).Y);
  btn.Down := False;
end;

procedure TCCDDBNavigator.Loaded;
begin
  inherited;
//  DataChanged;
end;

// Carga las imgenes de los ImageList en los botones
procedure TCCDDBNavigator.LoadImages;
  procedure LoadImg(image : Integer; btn : TButtonType);
  var
    b1, b2, b3 : TBitmap;
    rd, rs : TRect;
    loadb1, loadb2 : Boolean;

  begin
    if image < 0 then
      Exit;
    loadb1 := False;
    loadb2 := False;

    b1 := TBitmap.Create;
    b2 := TBitmap.Create;
    b1.PixelFormat := pf8bit;
    b2.PixelFormat := pf8bit;
    if Assigned(Images) and (Images.Count > image) then
    begin
      Images.GetBitmap(image, b1);
      loadb1 := True;
    end;
    if Assigned(DisabledImages) and (DisabledImages.Count > image) then
    begin
      DisabledImages.GetBitmap(image, b2);
      loadb2 := True;
    end;
    if loadb1 and loadb2 then
    begin
      b3 := TBitmap.Create;
      b3.PixelFormat := pf8bit;
      b3.Width := b1.Width + b2.Width;
      b3.Height := b1.Height;

      rs.Left := 0;
      rs.Top := 0;
      rs.Right := b1.Width;
      rs.Bottom := b1.Height;

      rd.Left := 0;
      rd.Top := 0;
      rd.Right := b1.Width;
      rd.Bottom := b1.Height;
      b3.Canvas.CopyRect(rd, b1.Canvas, rs);

      rd.Left := b1.Width;
      rd.Right := b1.Width * 2;
      b3.Canvas.CopyRect(rd, b2.Canvas, rs);

      btn.Glyph.Assign(b3);
      btn.NumGlyphs := 2;
      b3.Free;
    end
    else if loadb1 then
    begin
      btn.Glyph.Assign(b1);
      btn.NumGlyphs := 1;
    end
    else if loadb2 then
    begin
      btn.Glyph.Assign(b2);
      btn.NumGlyphs := 1;
    end;

    b2.Free;
    b1.Free;
  end;

var
  i : TButtonsTypes;
begin
  if not ImagesLoaded then
  begin
    for i := lbnSearch to lbnRefresh do
      if FButtons[i] <> nil then
        LoadImg(Ord(i), FButtons[i]);
    ImagesLoaded := True;
  end;

end;

procedure TCCDDBNavigator.SetEnabled(Value: Boolean);
begin
  FEnabled := Value;
  inherited SetEnabled(Value);
  DataChanged;
end;

{ TNavDataLink }

procedure TNavDataLink.ActiveChanged;
begin
  if FNavigator <> nil then FNavigator.ActiveChanged;
end;

constructor TNavDataLink.Create(Nav: TCCDDBNavigator);
begin
  inherited Create;
  FNavigator := Nav;
end;

procedure TNavDataLink.DataSetChanged;
begin
  if FNavigator <> nil then FNavigator.DataChanged;
end;

destructor TNavDataLink.Destroy;
begin
  FNavigator := nil;
  inherited Destroy;
end;

procedure TNavDataLink.EditingChanged;
begin
  if FNavigator <> nil then FNavigator.EditingChanged;
end;

end.
