{******************************************************************************}
{  DBImageGrid 1.0                                                        }
{  Autor:  Luis Bataller - Club Delphi                              }
{  Pgina de descargas: http://www.clubdelphi.com/componentes/                 }
{  Pas de orgen: Espaa, agosto de 2007.                                 }
{                                                                              }
{  TCCDDBImageGrid                                                           }
{                                                                              }
{  Componente de acceso a base de datos que muestra una imagen y/o un texto    }
{  por cada registro al estilo del DBCtrlGrid. Permite la navegacin a travs }
{  de las pginas segn controles personalizados.                              }
{  autocompletar.                                                              }
{  Permite cambiar el orden de los registros segn un campo de ordenacin.   }
{                                                                              }
{******************************************************************************}
unit UCCDDBImageGrid;

interface

uses
  SysUtils, Windows, Messages, Classes,
  Graphics, Controls, Forms, Dialogs,
  DB, DBGrids, stdCtrls,
  jpeg, variants, contnrs, extCtrls, Buttons;


type
  // Muestra los datos a travs de filas o columnas
  TbgOrientation = (bgHorizontal, bgVertical);
  // Posicin de los controles de navegacin
  TbgPosNavigator = (bgNone, bgTopLeft, bgTopRight, bgBottomLeft, bgBottomRight,
                     bgTop, bgLeft, bgRight, bgBottom);
  // Mtodo para mostrar las imgenes
  TbgDisplayMode = (bgCopy, bgStretch, bgZoom);

  TCCDDBImageGrid = class;

  // Enlace del control con el dataset que contiene los datos
  TCCDDBImageGridLink = class(TDataLink)
  private
    FButtonGrid: TCCDDBImageGrid;
  protected
    procedure ActiveChanged; override;
    procedure DataSetChanged; override;
    procedure DataSetScrolled(distance : integer); override;
    function IsDataSetAssigned : boolean;
    function IsDataSetActive : Boolean;
  public
    constructor Create(ButtonGrid: TCCDDBImageGrid);
    destructor Destroy; override;
  end;

  // Clase principal del componente
  TCCDDBImageGrid = class(TCustomControl)
  private
    FRepaint : Boolean;
    FPage : integer;
    FDataLink : TCCDDBImageGridLink;
    FColCount: integer;
    FRowCount: integer;
    FTextField: String;
    FImageField: String;
    FDataSource: TDataSource;
    FOrientation: TbgOrientation;
    FBrush: TBrush;
    FFont: TFont;
    FPen: TPen;
    FRelation : extended;
    FTextHeight: string;
    FPosNavigator: TbgPosNavigator;
    FNumRegs : integer;
    FDisplayMode: TbgDisplayMode;
    FNextPageControl: TControl;
    FPrevPageControl: TControl;
    FonClick: TNotifyEvent;
    FAlterFont: TFont;
    FOnKeyPress: TKeyPressEvent;
    Fvisible: boolean;
    FColorField: String;
    FKeyField: String;
    FImageCache: integer;
    FCache : TObjectList;
    FKeyCache : TStringList;
    FAllowChangeOrder: Boolean;
    FOrderField: String;
    TimerDragNext, TimerDragPrev : TTimer;
    DragRecordPos : Integer;
    DragRecordKey : string;
    procedure SetColCount(const Value: integer);
    procedure SetRowCount(const Value: integer);
    procedure SetDataSource(const Value: TDataSource);
    procedure SetImageField(const Value: String);
    procedure SetTextField(const Value: String);
    procedure DataSetChanged;
    procedure SetOrientation(const Value: TbgOrientation);
    procedure SetBrush(const Value: TBrush);
    procedure SetFont(const Value: TFont);
    procedure SetPen(const Value: TPen);
    procedure RepaintRequest(sender : TObject);
    procedure SetTextHeight(const Value: string);
    procedure SetPosNavigator(const Value: TbgPosNavigator);
    function CanPaint(x, y : integer) : boolean;
    procedure CalcNumRegs;
    procedure SetDisplayMode(const Value: TbgDisplayMode);
    procedure SetNextPageControl(const Value: TControl);
    procedure SetPrevPageControl(const Value: TControl);
    function RegAtPos(p : TPoint) : integer;
    procedure SetonClick(const Value: TNotifyEvent);
    procedure SetAlterFont(const Value: TFont);
    procedure SetOnKeyPress(const Value: TKeyPressEvent);
    procedure Setvisible(const Value: boolean);
    procedure SetColorField(const Value: String);
    procedure SetKeyField(const Value: String);
    procedure SetImageCache(const Value: integer);
    procedure SetAllowChangeOrder(const Value: Boolean);
    procedure SetOrderField(const Value: String);
    procedure ChangeActiveRecord(Value : Integer);
    procedure MoveFirstPageRecord;
    procedure DoTimerNext(Sender : TObject);
    procedure DoTimerPrev(Sender : TObject);

    { Private declarations }
  protected
    { Protected declarations }
    procedure Paint; override;
    procedure MouseDown(Button : TMouseButton; Shift : TShiftState;
                        X, Y : integer); override;
    procedure KeyPress(var Key : char); override;
    procedure CreateWnd; override;
    procedure Notification(AComponent: TComponent;
                           Operation: TOperation); override;

    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure DragOver(Source: TObject; X, Y: Integer; State: TDragState;
      var Accept: Boolean); override;
    procedure DoEndDrag(Target: TObject; X, Y: Integer); override;
    procedure DragOverPrev(Source: TObject; X, Y: Integer;
                           State: TDragState; var Accept: Boolean);
    procedure DragOverNext(Source: TObject; X, Y: Integer;
                           State: TDragState; var Accept: Boolean);
    function CheckDataSet : Boolean;
    function CheckDataSetActive : Boolean;
  public
    { Public declarations }
    inScroll : boolean;
    constructor Create (AOwner: TComponent); override;
    destructor Destroy; override;
    procedure NextPage;
    procedure PrevPage;
    procedure ReloadPage;
    procedure Scroll(distance : integer);
    procedure FirstPage;
  published
    { Published declarations }
    property width default 100;
    property height default 100;
    property left;
    property top;
    property visible : boolean read Fvisible write Setvisible;
    // Nmero de filas
    property RowCount : integer read FRowCount write SetRowCount;
    // Nmero de columnas
    property ColCount : integer read FColCount write SetColCount;
    property DataSource : TDataSource read FDataSource write SetDataSource;
    // Campo que contiene el texto a mostrar
    property TextField : String read FTextField write SetTextField;
    // Campo de la imagen
    property ImageField : String read FImageField write SetImageField;
    // Campo clave de la tabla
    property KeyField : String read FKeyField write SetKeyField;
    // Campo de ordenacin
    property OrderField : String read FOrderField write SetOrderField;
    // Campo con color personalizado para el fondo
    property ColorField : String read FColorField write SetColorField;
    // Orden por filas o columnas
    property Orientation : TbgOrientation read FOrientation write SetOrientation;
    property Pen : TPen read FPen write SetPen;
    property Brush : TBrush read FBrush write SetBrush;
    property Font : TFont read FFont write SetFont;
    // Tipo de letra en caso de que no exista imagen en el registro
    property AlterFont : TFont read FAlterFont write SetAlterFont;
    // Altura reservada para el texto (Los valores posibles estn en SetTextHeight)
    property TextHeight : string read FTextHeight write SetTextHeight;
    // Posicin donde se movern los controles para la navegacin
    property PosNavigator : TbgPosNavigator read FPosNavigator write SetPosNavigator;
    // Modo de ajuste de la imagen
    property DisplayMode : TbgDisplayMode read FDisplayMode write SetDisplayMode;
    // Control para la navegacin a la siguiente pgina
    property NextPageControl : TControl read FNextPageControl write SetNextPageControl;
    // Control para la navegacin a la pgina anterior
    property PrevPageControl : TControl read FPrevPageControl write SetPrevPageControl;
    // Tamao del cach de las imgenes (nmero de registros)
    property ImageCache : integer read FImageCache write SetImageCache;
    // Permitir el cambio del rden de presentacin de los registros
    property AllowChangeOrder : Boolean read FAllowChangeOrder write SetAllowChangeOrder;
    property Anchors;
    property OnKeyPress: TKeyPressEvent read FOnKeyPress write SetOnKeyPress;
    property OnClick : TNotifyEvent read FonClick write SetonClick;
  end;


implementation

{ TCCDDBImageGrid }

// Calcula el nmero de registros que se presentarn en cada pgina y
// establece el buffer del enlace a la tabla
procedure TCCDDBImageGrid.CalcNumRegs;
begin
  Case FPosNavigator of
    bgNone : FNumRegs := FRowCount * FColCount;
    bgTop, bgLeft, bgRight, bgBottom : FNumRegs := FRowCount * FColCount - 2;
    bgTopLeft, bgBottomLeft, bgTopRight, bgBottomRight : FNumRegs := FRowCount * FColCount - 1;
  end;
  if CheckDataSetActive then
  begin
    if FPage * FNumRegs < DataSource.DataSet.RecordCount then
      FDataLink.BufferCount := FNumRegs
    else
      FDataLink.BufferCount := FNumRegs - (FPage * FNumRegs - DataSource.DataSet.RecordCount);
  end;

end;

// Comprueba si una celda se debe rellenar con un registro o si pertenece
// a la navegacin
function TCCDDBImageGrid.CanPaint(x, y : integer): boolean;
begin
  if (x = 0) and (y = 0) then // left top
    CanPaint := (FPosNavigator = bgNone) or (FPosNavigator = bgRight) or
                (FPosNavigator = bgBottom) or (FPosNavigator = bgBottomRight) or
                (FPosNavigator = bgTopRight) or (FPosNavigator = bgBottomLeft)
  else if (x = 0) and (y = FRowCount - 1) then //left bottom
    CanPaint := (FPosNavigator = bgNone) or (FPosNavigator = bgRight) or
                (FPosNavigator = bgTop) or (FPosNavigator = bgTopLeft) or
                (FPosNavigator = bgTopRight) or (FPosNavigator = bgBottomRight)
  else if (x = FColCount - 1) and (y = 0) then //right top
    CanPaint := (FPosNavigator = bgNone) or (FPosNavigator = bgLeft) or
                (FPosNavigator = bgBottom) or (FPosNavigator = bgTopLeft) or
                (FPosNavigator = bgBottomRight) or (FPosNavigator = bgBottomLeft)
  else if (x = FColCount - 1) and (y = FRowCount - 1) then // right bottom
    CanPaint := (FPosNavigator = bgNone) or (FPosNavigator = bgLeft) or
                (FPosNavigator = bgTop) or (FPosNavigator = bgTopLeft) or
                (FPosNavigator = bgTopRight) or (FPosNavigator = bgBottomLeft)
  else
    CanPaint := True;
end;


// Constructor
constructor TCCDDBImageGrid.Create(AOwner: TComponent);
begin
  FDataLink := nil;
  FCache := TObjectList.Create;
  FKeyCache := TStringList.Create;
  FImageCache := 0;
  FRepaint := False;
  inherited Create(AOwner);

  // Timer para arrastrar y soltar en la ordenacin manual
  TimerDragNext := TTimer.Create(self);
  TimerDragNext.Enabled := False;
  TimerDragNext.Interval := 500;
  TimerDragNext.OnTimer := DoTimerNext;
  TimerDragPrev := TTimer.Create(self);
  TimerDragPrev.Enabled := False;
  TimerDragPrev.Interval := 500;
  TimerDragPrev.OnTimer := DoTimerPrev;

  inScroll := false;
  FDataLink := TCCDDBImageGridLink.Create(Self);

  // Valores por defecto
  FRowCount := 3;
  FColCount := 3;
  FPosNavigator := bgNone;

  FBrush := TBrush.Create;
  FPen := TPen.Create;
  FFont := TFont.Create;
  FAlterFont := TFont.Create;
  FBrush.OnChange := RepaintRequest;
  FPen.OnChange := RepaintRequest;
  FFont.OnChange := RepaintRequest;
  FAlterFont.OnChange := RepaintRequest;
  TextHeight := '1/3';
  FDisplayMode := bgStretch;
  FPage := 1;
  Visible := True;


end;

procedure TCCDDBImageGrid.DataSetChanged;
begin
  if not inScroll and CheckDataSetActive then
  begin
    if DataSource.DataSet.Eof then
      Scroll(DataSource.DataSet.RecordCount);
    if DataSource.DataSet.Bof then
      Scroll(-DataSource.DataSet.RecordCount);
  end;
  Paint;
end;

destructor TCCDDBImageGrid.Destroy;
begin
  TimerDragNext.Free;
  TimerDragPrev.Free;
  FBrush.Free;
  FPen.Free;
  FFont.Free;
  FAlterFont.Free;
  FCache.Free;
  FKeyCache.Free;
  if FDataLink <> nil then
    FDataLink.Free;
  inherited;
end;



procedure TCCDDBImageGrid.Paint;
var
  CurrentRecord1 : integer;


  // Mostrar el navegador
  procedure PaintArrows;
  var
    h : integer;
  begin
    h := height div FRowCount;
    case FPosNavigator of
      bgNone : begin
                 // En un principio nada
               end;
      bgLeft : begin
                 if Assigned(PrevPageControl) then
                 begin
                   FPrevPageControl.Width := width div FColCount;
                   FPrevPageControl.Height := Height div FRowCount;
                   FPrevPageControl.Left := Left;
                   FPrevPageControl.Top := Top;
                 end;
                 if Assigned(NextPageControl) then
                 begin
                   FNextPageControl.Width := width div FColCount;
                   FNextPageControl.Height := Height div FRowCount;
                   FNextPageControl.Left := Left;
                   FNextPageControl.Top := Top + height - height div rowcount;
                 end;
               end;
      bgRight : begin
                 if Assigned(PrevPageControl) then
                 begin
                   FPrevPageControl.Width := width div FColCount;
                   FPrevPageControl.Height := Height div FRowCount;
                   FPrevPageControl.Left := Left + width - width div colcount;
                   FPrevPageControl.Top := Top;
                 end;
                 if Assigned(NextPageControl) then
                 begin
                   FNextPageControl.Width := width div FColCount;
                   FNextPageControl.Height := Height div FRowCount;
                   FNextPageControl.Left := Left + width - width div colcount;
                   FNextPageControl.Top := Top + height - height div rowcount;
                 end;
               end;
      bgTop : begin
                 if Assigned(PrevPageControl) then
                 begin
                   FPrevPageControl.Width := width div FColCount;
                   FPrevPageControl.Height := Height div FRowCount;
                   FPrevPageControl.Left := Left;
                   FPrevPageControl.Top := Top;
                 end;
                 if Assigned(NextPageControl) then
                 begin
                   FNextPageControl.Width := width div FColCount;
                   FNextPageControl.Height := Height div FRowCount;
                   FNextPageControl.Left := Left + width - width div colcount;
                   FNextPageControl.Top := Top;
                 end;
               end;
      bgBottom : begin
                 if Assigned(PrevPageControl) then
                 begin
                   FPrevPageControl.Width := width div FColCount;
                   FPrevPageControl.Height := Height div FRowCount;
                   FPrevPageControl.Left := Left;
                   FPrevPageControl.Top := Top + height - height div rowcount;
                 end;
                 if Assigned(NextPageControl) then
                 begin
                   FNextPageControl.Width := width div FColCount;
                   FNextPageControl.Height := Height div FRowCount;
                   FNextPageControl.Left := Left + width - width div colcount;
                   FNextPageControl.Top := Top + height - height div rowcount;
                 end;
               end;
      bgTopLeft : begin
                 if Assigned(PrevPageControl) then
                 begin
                   FPrevPageControl.Width := width div FColCount;
                   if not Assigned(NextPageControl) then
                     FPrevPageControl.Height := Height div FRowCount;
                   FPrevPageControl.Left := Left;
                   FPrevPageControl.Top := Top;
                 end;
                 if Assigned(NextPageControl) then
                 begin
                   FNextPageControl.Width := width div FColCount;
                   if Assigned(PrevPageControl) then
                   begin
                     FPrevPageControl.Height := h div 2;
                     FNextPageControl.Height := h div 2;
                     FNextPageControl.Top := FPrevPageControl.Top + h div 2;
                     FNextPageControl.Left := left;
                   end
                   else begin
                     FNextPageControl.Height := Height div FRowCount;
                     FNextPageControl.Top := Top;
                     FNextPageControl.Left := left;
                   end;
                 end;
               end;
      bgTopRight : begin
                 if Assigned(PrevPageControl) then
                 begin
                   FPrevPageControl.Width := width div FColCount;
                   if not Assigned(NextPageControl) then
                     FPrevPageControl.Height := Height div FRowCount;
                   FPrevPageControl.Left := Left  + width - width div colcount;
                   FPrevPageControl.Top := Top;
                 end;
                 if Assigned(NextPageControl) then
                 begin
                   FNextPageControl.Width := width div FColCount;
                   if Assigned(PrevPageControl) then
                   begin
                     FPrevPageControl.Height := h div 2;
                     FNextPageControl.Height := h div 2;
                     FNextPageControl.Top := FPrevPageControl.Top + h div 2;
                     FNextPageControl.Left := Left  + width - width div colcount;
                   end
                   else begin
                     FNextPageControl.Height := Height div FRowCount;
                     FNextPageControl.Top := Top;
                     FNextPageControl.Left := Left  + width - width div colcount;
                   end;
                 end;
               end;
      bgBottomLeft : begin
                 if Assigned(PrevPageControl) then
                 begin
                   FPrevPageControl.Width := width div FColCount;
                   if not Assigned(NextPageControl) then
                     FPrevPageControl.Height := Height div FRowCount;
                   FPrevPageControl.Left := Left;
                   FPrevPageControl.Top := Top + height - height div rowcount;
                 end;
                 if Assigned(NextPageControl) then
                 begin
                   FNextPageControl.Width := width div FColCount;
                   if Assigned(PrevPageControl) then
                   begin
                     FPrevPageControl.Height := h div 2;
                     FNextPageControl.Height := h div 2;
                     FNextPageControl.Top := FPrevPageControl.Top + h div 2;
                     FNextPageControl.Left := Left;
                   end
                   else begin
                     FNextPageControl.Height := Height div FRowCount;
                     FNextPageControl.Top := Top;
                     FNextPageControl.Left := Left;
                   end;
                 end;
               end;
      bgBottomRight : begin
                 if Assigned(PrevPageControl) then
                 begin
                   FPrevPageControl.Width := width div FColCount;
                   if not Assigned(NextPageControl) then
                     FPrevPageControl.Height := Height div FRowCount;
                   FPrevPageControl.Left := Left + width - width div colcount;
                   FPrevPageControl.Top := Top + height - height div rowcount;
                 end;
                 if Assigned(NextPageControl) then
                 begin
                   FNextPageControl.Width := width div FColCount;
                   if Assigned(PrevPageControl) then
                   begin
                     FPrevPageControl.Height := h div 2;
                     FNextPageControl.Height := h div 2;
                     FNextPageControl.Top := FPrevPageControl.Top + h div 2;
                     FNextPageControl.Left := Left + width - width div colcount;
                   end
                   else begin
                     FNextPageControl.Height := Height div FRowCount;
                     FNextPageControl.Top := Top + height - height div rowcount;
                     FNextPageControl.Left := Left + width - width div colcount;
                   end;
                 end;
               end;
    end;
    if FPosNavigator <> bgNone then
    begin
      Canvas.Brush.Color := Brush.Color;
      Canvas.Font.Color := Font.Color;
      Canvas.Font.Size := Font.Size;
      Canvas.Pen.Color := Pen.Color;
      if Assigned(PrevPageControl) then
      begin
        Canvas.Rectangle(FPrevPageControl.Left - Left, FPrevPageControl.Top - Top,
                         FPrevPageControl.Left - Left + FPrevPageControl.Width,
                         FPrevPageControl.Top - Top + FPrevPageControl.Height);
        Canvas.TextOut(FPrevPageControl.Left - Left + 2,
                       FPrevPageControl.Top - Top + Canvas.TextHeight('Anterior') + 2, 'Anterior');
      end;
      if Assigned(NextPageControl) then
      begin
        Canvas.Rectangle(FNextPageControl.Left - Left, FNextPageControl.Top - Top,
                         FNextPageControl.Left - Left + FNextPageControl.Width,
                         FNextPageControl.Top - Top + FNextPageControl.Height);
        Canvas.TextOut(FNextPageControl.Left - Left + 2,
                       FNextPageControl.Top - Top + Canvas.TextHeight('Siguiente') + 2, 'Siguiente');
      end;
    end;
    if FNextPageControl <> nil then
    begin
      FNextPageControl.Invalidate;
    end;
    if FPrevPageControl <> nil then
    begin
      FPrevPageControl.Invalidate;
    end;
  end;

  // Muestra las lneas separadoras de los registros
  procedure PaintPanels;
  var
    i, j : integer;
    w, h : integer;
    r : TRect;
    offsetX, offsetY : integer;
  begin
    r.Left := 0;
    r.Top := 0;
    r.Right := Width;
    r.Bottom := Height;
    canvas.Brush := FBrush;
    canvas.Pen := FPen;
    canvas.FillRect(r);
    w := width div FColCount;
    h := height div FRowCount;
    for i := 0 to FColCount - 1 do
      for j := 0 to FRowCount - 1 do
      begin
        if i = FColCount - 1 then
          offsetX := width
        else
          offsetX := (i + 1) * w;
        if j = FRowCount - 1 then
          offsetY := height
        else
          offsetY := (j + 1) * h;
        Canvas.Rectangle(i * w, j * h, OffsetX, OffsetY);
      end;
  end;

  // Muestra el texto los registros
  procedure PaintLabels;
  var
    w, h : integer;

    // Revisar: dependiendo del tamao de algunas palabras, puede mostrarse
    // con una esttica mala
    procedure PrintText(x, y : integer; s : string; AllSize : boolean = false);
    var
      st, stw : TStringList;
      i : integer;
      r : TRect;
      sizeant : Integer;
    begin

      if AllSize then
        canvas.Font := FAlterFont
      else
        canvas.Font := FFont;

      st := TStringList.Create;
      stw := TStringList.Create;
      stw.Delimiter := ' ';
      stw.DelimitedText := s;


      // separamos las palabras para hacer un 'wordWrap'
      st.Clear;
      i := 0;
      while stw.Count > 0 do
      begin
        st.Add(stw[0]);
        stw.Delete(0);
        while (stw.Count > 0) and
              ((w - 2) > canvas.TextWidth(st[i] + ' ' + stw[0])) do
        begin
          st[i] := st[i] + ' ' + stw[0];
          stw.Delete(0);
        end;
        i := i + 1;
      end;

      r.Left := x + 3;
      if AllSize then
        r.Top := y
      else
        r.Top := y + Trunc(h * (1 - FRelation));
      r.Right := x + w - 2;
      r.Bottom := y + h - 2;
      canvas.Brush.Style := bsClear;
      // Mostramos las palabras
      for i := 0 to st.count - 1 do
      begin
        // si la palabra no entra en el ancho de la celda, reducimos el tamao
        sizeAnt := Canvas.Font.Size;
        while (Canvas.TextWidth(st[i]) > w - 2) and (Canvas.Font.Size >= 8) do
          Canvas.Font.Size := Canvas.Font.Size - 1;
        if not AllSize then
          Canvas.TextRect(r, x + 1 + (w - canvas.TextWidth(st[i])) div 2,
                          y + Trunc(h * (1 - FRelation)) + i * canvas.TextHeight(st[i]), st[i])
        else
          Canvas.TextRect(r, x + 1 + (w - canvas.TextWidth(st[i])) div 2,
                          y + i * canvas.TextHeight(st[i]), st[i]);
        Canvas.Font.Size := sizeAnt;
      end;
      canvas.Brush.Style := FBrush.Style;
      st.Free;
      stw.Free;
    end;

  // Mostrar la imagen
  function drawImage(x, y : integer) : boolean;
    procedure CreateJPEG(var B : TGraphic);
    var
      buf : word;
      p : TPicture;
      MemBuffer: TMemoryStream;
    begin
        MemBuffer := TMemoryStream.Create;

        TBlobField(FDataLink.DataSet.FieldByName(ImageField)).SaveToStream(MemBuffer);
        MemBuffer.Position :=0;
        B:= nil;
        if MemBuffer.Size > SizeOf(buf) then
        begin
          MemBuffer.read(buf, SizeOf(buf));
          // escogemos el tipo de imagen (bmp, jpeg, icon...)
          case buf of
            $0000 : B := TIcon.Create;
            $0001 : begin
                      P := TPicture.Create;
                      p.Assign(FDataLink.DataSet.FieldByName(ImageField));
                      B := TBitmap.Create;
                      B.Assign(p);
                      p.Free;
                    end;
            $4D42 : B := TBitmap.Create;
            $CDD7 : B := TMetafile.Create;
            $D8FF : B := TJPEGImage.Create;
          end;
        end;

        MemBuffer.Position := 0;
        if (B <> nil) and (buf <> $0001) then
          try
             B.LoadFromStream(MemBuffer);
          except
            B.Free;
            B := nil;
          end;
        MemBuffer.Free;
    end;

  var
    B : TGraphic;
    r, r1 : TRect;
    bmp : TBitmap;
    scale1, scale2 : extended;
    offset : integer;
    pos : integer;

  begin
    if (ImageField <> '') and
       (not FDataLink.DataSet.FieldByName(ImageField).IsNull) then
    begin
      drawImage := true;
      // Primero comprobamos si la imagen est en el cach
      if (FImageCache > 0) and (FDataLink.DataSet.FindField(KeyField) <> nil) then
      begin
        pos := FKeyCache.IndexOf(FDataLink.DataSet.FieldByName(KeyField).AsString);
        if pos <> -1 then
          B := FCache[pos] as TGraphic
        else begin
          if FCache.Count >= FImageCache then
          begin
            FCache.Delete(0);
            FKeyCache.Delete(0);
          end;
          CreateJPEG(B);
          FCache.Add(B);
          FKeyCache.Add(FDataLink.DataSet.FieldByName(KeyField).AsString);
        end;
      end
      else CreateJPEG(B);
      if B <> nil then
      begin
        r.Left := x + 3;
        r.Top := y + 2;
        r.Right := x + w - 2;
        r.Bottom := y + trunc(h * (1 - FRelation));
        // Segn el modo de visualizacin...
        case DisplayMode of
          bgCopy : begin
                     bmp := TBitmap.Create;
                     r1.Left := 0;
                     r1.Top := 0;
                     r1.Right := w - 2;
                     r1.Bottom := trunc(h * (1 - FRelation));
                     bmp.Canvas.Draw(0, 0, B);
                     Canvas.CopyMode := cmSrcCopy;
                     Canvas.CopyRect(r, bmp.Canvas, r1);
                     bmp.Free;
                   end;
          bgStretch : Canvas.StretchDraw(r, B);
          bgZoom : begin
                     scale1 := (w - 2.0) / b.Width;
                     scale2 := h * (1 - FRelation) / b.Height;
                     if scale1 > scale2 then
                     begin
                       offset := (trunc((w - 2) * scale2) + (w - 2)) div 2 - trunc((w - 2) * scale2);
                       r.Left := x + offset;
                       r.Right := x + trunc((w - 2) * scale2) + offset;
                       Canvas.StretchDraw(r, B);
                     end
                     else
                     begin
                       r.Bottom := y + trunc(trunc(h * (1 - FRelation)) * scale1);
                       Canvas.StretchDraw(r, B);
                     end
                   end;
        end;
        if FImageCache <= 0 then
          B.Free;
      end;
    end
    else DrawImage := false;
  end;

  var
    i, j : integer;
    offset : integer;
    offsetX, offsetY : integer;

  begin
    canvas.Font := FFont;
    w := width div FColCount;
    h := height div FRowCount;

    // Recorremos el dataset y mostramos los registros

    offset := 0;
    if FOrientation = bgVertical then
    begin
      for i := 0 to FColCount - 1 do
        for j := 0 to FRowCount - 1 do
          if CheckDataSetActive then
          begin
            if canPaint(i, j) and ((i * FRowCount) + j - offset < FDataLink.BufferCount) then
            begin
              ChangeActiveRecord((i * FRowCount) + j - offset);

              if i = FColCount - 1 then
                offsetX := width
              else
                offsetX := (i + 1) * w;
              if j = FRowCount - 1 then
                offsetY := height
              else
                offsetY := (j + 1) * h;
              if ((FDataLink.DataSource.DataSet.FindField(ColorField) <> nil)) and CheckDataSetActive and
                 (Brush.Color <> FDataLink.DataSet.FieldByName(ColorField).AsInteger) then
              begin
                Canvas.Brush.Color := FDataLink.DataSet.fieldByName(ColorField).AsInteger;
                Canvas.Rectangle(i * w, j * h, OffsetX, OffsetY);
                Canvas.Brush.Color := FBrush.Color;
              end;

              if DrawImage(i * w, j * h) and (FDataLink.DataSource.DataSet.FindField(TextField) <> nil) then
                PrintText(i * w, j * h,
                        FDataLink.DataSet.FieldByName(FTextField).AsString)
              else if (FDataLink.DataSource.DataSet.FindField(TextField) <> nil) then
              begin
                PrintText(i * w, j * h,
                        FDataLink.DataSet.FieldByName(FTextField).AsString, true);
              end;
            end
            else
              offset := offset + 1;
          end;
    end
    else begin
      for i := 0 to FRowCount - 1 do
        for j := 0 to FColCount - 1 do
          if CheckDataSetActive then
          begin
            if canPaint(j, i) and ((i * FColCount) + j - offset < FDataLink.BufferCount) then
            begin
              ChangeActiveRecord((i * FColCount) + j - offset);

              if i = FRowCount - 1 then
                offsetY := height
              else
                offsetY := (i + 1) * h;
              if j = FColCount - 1 then
                offsetX := width
              else
                offsetX := (j + 1) * w;
              if ((FDataLink.DataSource.DataSet.FindField(ColorField) <> nil)) and CheckDataSetActive and
                 (Brush.Color <> FDataLink.DataSet.FieldByName(ColorField).AsInteger) then
              begin
                Canvas.Brush.Color := FDataLink.DataSet.fieldByName(ColorField).AsInteger;
                Canvas.Rectangle(j * w, i * h, OffsetX, OffsetY);
                Canvas.Brush.Color := FBrush.Color;
              end;
              if DrawImage(j * w, i * h) and ((FDataLink.DataSource.DataSet.FindField(TextField) <> nil)) then
                PrintText(j * w, trunc(i * h),
                        FDataLink.DataSet.FieldByName(FTextField).AsString)
              else if (FDataLink.DataSource.DataSet.FindField(TextField) <> nil) then
              begin
                PrintText(j * w, trunc(i * h),
                        FDataLink.DataSet.FieldByName(FTextField).AsString, true);
              end;
            end
            else
              offset := offset + 1;
          end;
     end;
    end;

begin
  if FRepaint and visible then
  begin
    PaintPanels;
    if CheckDataSetActive then
    begin
      InScroll := True;
      FRepaint := False;
      inherited;
      if DataSource.DataSet.Active then
      begin
        currentRecord1 := FDatalink.ActiveRecord;
        PaintLabels;
        ChangeActiveRecord(currentRecord1);
      end;
      PaintArrows;
      FRepaint := True;
      inScroll := False;
    end;
  end;
end;

// Refresh de la pgina
procedure TCCDDBImageGrid.ReloadPage;
begin
  if CheckDataSetActive then
  begin
    FRepaint := false;
    inScroll := true;
    DataSource.DataSet.First;
    CalcNumRegs;
    DataSource.DataSet.MoveBy(FPage * FNumregs - 1);
    CalcNumRegs;
    inScroll := false;
    FRepaint := true;
    Paint;
  end;
end;

// Siguiente pgina
procedure TCCDDBImageGrid.NextPage;
begin
  if CheckDataSetActive then
  begin
    FRepaint := false;
    inScroll := true;
    CalcNumRegs;
    if (FPage <= DataSource.DataSet.RecordCount div FNumRegs) then
      FPage := FPage + 1
    else
      FPage := 1;
    DataSource.DataSet.First;
    DataSource.DataSet.MoveBy(FPage * FNumregs - 1);
    CalcNumRegs;
    inScroll := false;
    FRepaint := true;
    Paint;
    MoveFirstPageRecord;
  end;
end;


// Pgina anterior
procedure TCCDDBImageGrid.PrevPage;
begin
  if CheckDataSetActive then
  begin
    FRepaint := false;
    inScroll := true;
    if (FPage > 1) then
      FPage := FPage - 1
    else if DataSource.DataSet.RecordCount mod FNumRegs = 0 then
      FPage := DataSource.DataSet.RecordCount div FNumRegs
    else
      FPage := DataSource.DataSet.RecordCount div FNumRegs + 1;
    DataSource.DataSet.First;
    CalcNumRegs;
    DataSource.DataSet.MoveBy(FPage * FNumregs - 1);
    CalcNumRegs;
    inScroll := false;
    FRepaint := true;
    Paint;
    MoveFirstPageRecord;
  end;
end;

procedure TCCDDBImageGrid.RepaintRequest(sender: TObject);
begin
  Paint;
end;


procedure TCCDDBImageGrid.SetBrush(const Value: TBrush);
begin
  FBrush.Assign(Value);
  Paint;
end;

procedure TCCDDBImageGrid.SetColCount(const Value: integer);
begin
  FColCount := Value;
  CalcNumRegs;
  Invalidate;
end;

procedure TCCDDBImageGrid.SetDataSource(const Value: TDataSource);
begin
  FDataSource := Value;
  FDataLink.DataSource := Value;
  CalcNumRegs;
  Paint;
end;

procedure TCCDDBImageGrid.SetDisplayMode(const Value: TbgDisplayMode);
begin
  FDisplayMode := Value;
  Paint;
end;

procedure TCCDDBImageGrid.SetFont(const Value: TFont);
begin
  FFont.Assign(Value);
  Paint;
end;

procedure TCCDDBImageGrid.SetImageField(const Value: String);
begin
  FImageField := Value;
  Paint;
end;

procedure TCCDDBImageGrid.SetNextPageControl(const Value: TControl);
begin
  FNextPageControl := Value;
  Paint;
end;

procedure TCCDDBImageGrid.SetOrientation(const Value: TbgOrientation);
begin
  FOrientation := Value;
  Paint;
end;

procedure TCCDDBImageGrid.SetPen(const Value: TPen);
begin
  FPen.Assign(Value);
  Paint;
end;

procedure TCCDDBImageGrid.SetPosNavigator(const Value: TbgPosNavigator);
begin
  FPosNavigator := Value;
  CalcNumRegs;
  Paint;
end;

procedure TCCDDBImageGrid.SetPrevPageControl(const Value: TControl);
begin
  FPrevPageControl := Value;
  Paint;
end;

procedure TCCDDBImageGrid.SetRowCount(const Value: integer);
begin
  FRowCount := Value;
  CalcNumRegs;
  Paint;
end;


procedure TCCDDBImageGrid.SetTextField(const Value: String);
begin
  FTextField := Value;
  Paint;
end;

procedure TCCDDBImageGrid.SetTextHeight(const Value: string);
begin
  FTextHeight := Value;
  if FTextHeight = '0' then
    FRelation := 0
  else if FTextHeight = '1/5' then
    FRelation := 0.20
  else if FTextHeight = '2/5' then
    FRelation := 0.40
  else if FTextHeight = '3/5' then
    FRelation := 0.60
  else if FTextHeight = '4/5' then
    FRelation := 0.80
  else if FTextHeight = '1/4' then
    FRelation := 0.25
  else if FTextHeight = '1/2' then
    FRelation := 0.50
  else if FTextHeight = '3/4' then
    FRelation := 0.75
  else if FTextHeight = '1/3' then
    FRelation := 0.3333
  else if FTextHeight = '2/3' then
    FRelation := 0.6666
  else if FTextHeight = '1' then
    FRelation := 1
  else begin
    FRelation := 0.6666;
    FTextHeight := '1/3';
  end;
  Paint;
end;



// Devuelve el nmero registro que se encuentra en un punto de la pantalla

function TCCDDBImageGrid.RegAtPos(p: TPoint): integer;
var
  i, j : integer;
  pos : integer;
  w, h: integer;
  encontrado : boolean;
begin
  pos := 0;
  encontrado := false;
  w := width div colCount;
  h := height div rowCount;
  if FOrientation = bgHorizontal then
  begin
    i := 0;
    while not encontrado and (i < FRowCount) do
    begin
      j := 0;
      while not encontrado and (j < FColCount) do
      begin
        if (p.X < (j + 1) * w) and (p.X > j * w) and
           (p.Y < (i + 1) * h) and (p.Y > i * h) then
           encontrado := true;
        if CanPaint(i, j) and not encontrado then
          pos := pos + 1;
        j := j + 1;
      end;
      i := i + 1;
    end;
  end
  else begin
    j := 0;
    while not encontrado and (j < FColCount) do
    begin
      i := 0;
      while not encontrado and (i < FRowCount) do
      begin
        if (p.X < (j + 1) * w) and (p.X > j * w) and
           (p.Y < (i + 1) * h) and (p.Y > i * h) then
           encontrado := true;
        if CanPaint(i, j) and not encontrado then
          pos := pos + 1;
        i := i + 1;
      end;
      j := j + 1;
    end;
  end;
  if not encontrado then
    pos := -1;
  RegAtPos := Pos;
end;

procedure TCCDDBImageGrid.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: integer);
var
  aPoint : TPoint;
  pos : integer;
begin
  inherited;
  // si estamos en modo de cambio de ordenacin comenzamos a arrastrar
  if FAllowChangeOrder and CheckDataSetActive then
  begin
    BeginDrag(False);
    if FNextPageControl <> nil then
      FNextPageControl.Visible := False;
    if FPrevPageControl <> nil then
      FPrevPageControl.Visible := False;
  end;
  // al pulsar el ratn, nos posicionamos en el registro donde se ha pulsado
  if CheckDataSetActive then
  begin
    FRepaint := false;
    InScroll := true;
    aPoint.X := X;
    aPoint.Y := Y;
    pos := RegAtPos(aPoint);
    if pos <> -1 then
    begin

      if (FPage > DataSource.DataSet.RecordCount div FNumRegs) then
      begin
        FDataLink.DataSet.Last;
        FDataLink.DataSet.MoveBy(-(FDataLink.BufferCount - pos - 1));
      end
      else begin
        ChangeActiveRecord(0);
        if pos = 0 then
        begin
          FDataLink.DataSet.MoveBy(1);
          FDataLink.DataSet.MoveBy(-1);
        end
        else
          FDataLink.DataSet.MoveBy(pos);
      end;
    end;
    InScroll := false;
    setFocus;
    FRepaint := true;
    if Self.Dragging and (DataSource.DataSet.FindField(KeyField) <> nil) and
       (DataSource.DataSet.FindField(OrderField) <> nil) then
    begin
      DragRecordPos := DataSource.DataSet.FieldByName(OrderField).AsInteger;
      DragRecordKey := DataSource.DataSet.FieldByName(KeyField).AsString;
    end;
    if Assigned(FonClick) then
      FonClick(self);
  end;
end;


// Scroll, se mueve la distancia hacia adelante (o hacia atras si es negativo)
// el valor de distance
procedure TCCDDBImageGrid.Scroll(distance : integer);
  function FindKey(key : variant) : boolean;
  begin
    FindKey := DataSource.DataSet.Locate(FKeyField, key, []);
  end;

var
  key : variant;
begin
  if CheckDataSetActive and
     (DataSource.DataSet.FindField(KeyField) <> nil) then
  begin
    inScroll := True;
    FRepaint := False;
    key := FDataLink.DataSet.FieldByName(FKeyField).AsVariant;
    if (distance <> 0) then
    begin
      if distance > 0 then
        FPage := FPage + 1 + Distance div FNumRegs
      else if distance < 0 then
        FPage := FPage - 1 + Distance div FNumRegs;
      if FPage < 1 then
        FPage := 1;
      if FPage > DataSource.DataSet.RecordCount div FNumRegs + 1 then
        FPage := DataSource.DataSet.RecordCount div FNumRegs + 1;
      calcNumRegs;
      ReloadPage;
      FindKey(key);
    end;
    FRepaint := True;
    inScroll := False;
  end;
end;

procedure TCCDDBImageGrid.SetonClick(const Value: TNotifyEvent);
begin
  FonClick := Value;
end;

procedure TCCDDBImageGrid.SetAlterFont(const Value: TFont);
begin
  FAlterFont.Assign(Value);
  Paint;
end;

procedure TCCDDBImageGrid.KeyPress(var Key: char);
begin
  inherited;
  if Assigned(FOnKeyPress) then
    FOnKeyPress(self, Key);

end;

procedure TCCDDBImageGrid.SetOnKeyPress(const Value: TKeyPressEvent);
begin
  FOnKeyPress := Value;
end;

// Mueve a la primera pgina
procedure TCCDDBImageGrid.FirstPage;
begin
  CalcNumRegs;
  FPage := 0;
  NextPage;
  Invalidate;
end;

procedure TCCDDBImageGrid.Setvisible(const Value: boolean);
begin
  Fvisible := Value;
  Inherited Visible := Value;
  if Assigned(FPrevPageControl) then
    FPrevPageControl.Visible := Value;
  if Assigned(FNextPageControl) then
    FNextPageControl.Visible := Value;
end;

procedure TCCDDBImageGrid.SetColorField(const Value: String);
begin
  FColorField := Value;
  Paint;
end;

procedure TCCDDBImageGrid.SetKeyField(const Value: String);
begin
  FKeyField := Value;
end;

procedure TCCDDBImageGrid.SetImageCache(const Value: integer);
begin
  FImageCache := Value;
end;

function TCCDDBImageGrid.CheckDataSet: Boolean;
begin
  Result := Assigned(DataSource) and Assigned(DataSource.DataSet) and
            (FDataLink <> nil);
end;

function TCCDDBImageGrid.CheckDataSetActive: Boolean;
begin
  Result := Assigned(DataSource) and Assigned(DataSource.DataSet) and
            (FDataLink <> nil) and DataSource.DataSet.Active;
end;

procedure TCCDDBImageGrid.CreateWnd;
begin
  inherited;
  FRepaint := True;
  FirstPage;
  Paint;
end;

procedure TCCDDBImageGrid.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  // Por ahora nada
  inherited Notification(AComponent, Operation);
end;

procedure TCCDDBImageGrid.SetAllowChangeOrder(const Value: Boolean);
begin
  FAllowChangeOrder := Value;
  Self.DragMode := dmManual;
end;

// Finaliza el arrastre del registro a su nueva posicin
procedure TCCDDBImageGrid.DoEndDrag(Target: TObject; X, Y: Integer);
var
  rec1 : Variant;
  pos1, pos2 : Variant;
  nRec : Integer;
  aPoint : TPoint;
begin
  if CheckDataSetActive and
    (DataSource.DataSet.FindField(KeyField) <> nil) and
    (DataSource.DataSet.FindField(OrderField) <> nil) then
  begin
    FRepaint := False;
    inScroll := True;
    pos1 := DragRecordPos;
    rec1 := DragRecordKey;

    aPoint.X := x;
    aPoint.Y := y;
    nRec := RegAtPos(aPoint);
    if nRec <> -1 then
    begin
      FDataLink.DataSet.MoveBy(-FDataLink.ActiveRecord);
      FDataLink.DataSet.MoveBy(nRec);
      pos2 := DataSource.DataSet.FieldByName(OrderField).AsVariant;

      if not (DataSource.DataSet.State in [dsEdit, dsInsert]) then
        DataSource.DataSet.Edit;
      DataSource.DataSet.FieldByName(OrderField).AsVariant := pos1;
      DataSource.DataSet.Post;
      if DataSource.DataSet.Locate(KeyField, rec1, []) then
      begin
        if not (DataSource.DataSet.State in [dsEdit, dsInsert]) then
          DataSource.DataSet.Edit;
        DataSource.DataSet.FieldByName(OrderField).AsVariant := pos2;
        DataSource.DataSet.Post;
      end;
      FDataLink.DataSet.MoveBy(-FDataLink.ActiveRecord);
      FDataLink.DataSet.MoveBy(nRec);
      ReloadPage;
    end;
    inScroll := False;
    FRepaint := True;
  end;
  if FNextPageControl <> nil then
    FNextPageControl.Visible := True;
  if FPrevPageControl <> nil then
    FPrevPageControl.Visible := True;
end;

procedure TCCDDBImageGrid.DragOver(Source: TObject; X, Y: Integer;
  State: TDragState; var Accept: Boolean);
begin
  inherited;
  Accept := Source = Self;
  if Accept then
  begin
    if Assigned(NextPageControl) and (x > FNextPageControl.Left - Left) and
       (x < FNextPageControl.Left - Left + FNextPageControl.Width) and
       (y > FNextPageControl.Top - Top) and
       (y < FNextPageControl.Top - Top + FNextPageControl.Top) then
       Self.DragOverNext(Source, x, y, state, accept)
    else if Assigned(PrevPageControl) and (x > FPrevPageControl.Left - Left) and
       (x < FPrevPageControl.Left - Left + FPrevPageControl.Width) and
       (y > FPrevPageControl.Top - Top) and
       (y < FPrevPageControl.Top - Top + FPrevPageControl.Top) then
       Self.DragOverPrev(Source, x, y, state, accept);
  end;

end;

procedure TCCDDBImageGrid.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
  inherited;

end;

procedure TCCDDBImageGrid.SetOrderField(const Value: String);
begin
  FOrderField := Value;
end;

procedure TCCDDBImageGrid.ChangeActiveRecord(Value: Integer);
begin
  if CheckDataSetActive then
  begin
    FRepaint := False;
    FDataLink.ActiveRecord := Value;
    FRepaint := True;
  end;
end;

procedure TCCDDBImageGrid.MoveFirstPageRecord;
begin
  if CheckDataSetActive then
    FDataLink.MoveBy(-FDataLink.ActiveRecord);
end;

// Para mover el registro a la siguiente pgina
procedure TCCDDBImageGrid.DragOverNext(Source: TObject; X, Y: Integer;
  State: TDragState; var Accept: Boolean);
begin
  if Source = Self then
  begin
    Accept := True;
    if not TimerDragNext.Enabled then
    begin
      TimerDragNext.Enabled := True;
      TimerDragPrev.Enabled := False;
    end;
  end;
end;

// para mover el registro a la pgina anterior
procedure TCCDDBImageGrid.DragOverPrev(Source: TObject; X, Y: Integer;
  State: TDragState; var Accept: Boolean);
begin
  if Source = Self then
  begin
    Accept := True;
    if not TimerDragPrev.Enabled then
    begin
      TimerDragNext.Enabled := False;
      TimerDragPrev.Enabled := True;
    end;
  end;
end;

procedure TCCDDBImageGrid.DoTimerNext(Sender: TObject);
var
  x, y : Integer;
begin
  x := ScreenToClient(Mouse.CursorPos).X;
  y := ScreenToClient(Mouse.CursorPos).Y;
  if (x > FNextPageControl.Left - Left) and
     (x < FNextPageControl.Left - Left + FNextPageControl.Width) and
     (y > FNextPageControl.Top - Top) and
     (y < FNextPageControl.Top - Top + FNextPageControl.Top) then
    NextPage;
  TimerDragNext.Enabled := False;
end;

procedure TCCDDBImageGrid.DoTimerPrev(Sender: TObject);
var
  x, y : Integer;
begin
  x := ScreenToClient(Mouse.CursorPos).X;
  y := ScreenToClient(Mouse.CursorPos).Y;
  if (x > FPrevPageControl.Left - Left) and
     (x < FPrevPageControl.Left - Left + FPrevPageControl.Width) and
     (y > FPrevPageControl.Top - Top) and
     (y < FPrevPageControl.Top - Top + FPrevPageControl.Top) then
    PrevPage;
  TimerDragPrev.Enabled := False;
end;


{ TCCDDBImageGridLink }

procedure TCCDDBImageGridLink.ActiveChanged;
begin
  inherited;
  if (FButtonGrid <> nil) and IsDataSetAssigned then
    FButtonGrid.DataSetChanged;
  if IsDataSetActive then
    FButtonGrid.FirstPage;
end;

constructor TCCDDBImageGridLink.Create(ButtonGrid: TCCDDBImageGrid);
begin
  FButtonGrid := nil;
  inherited Create;
  FButtonGrid := ButtonGrid;
  VisualControl := True;
  DataSource := ButtonGrid.DataSource;
  RPR;
end;

procedure TCCDDBImageGridLink.DataSetChanged;
begin
  inherited;
  if (FButtonGrid <> nil) and IsDataSetActive then
    FButtonGrid.DataSetChanged;
end;

procedure TCCDDBImageGridLink.DataSetScrolled(distance : integer);
begin
  if (FButtonGrid <> nil) and not FButtonGrid.inScroll and
     IsDataSetActive then
  begin
    FButtonGrid.Scroll(distance);
  end;
  inherited;
end;

destructor TCCDDBImageGridLink.Destroy;
begin
  FButtonGrid := nil;
  inherited;
end;

function TCCDDBImageGridLink.IsDataSetActive: Boolean;
begin
  Result := (FButtonGrid <> nil) and Assigned(FButtonGrid.DataSource) and
            Assigned(FButtonGrid.DataSource.DataSet) and
            FButtonGrid.DataSource.DataSet.Active;
end;

function TCCDDBImageGridLink.IsDataSetAssigned: boolean;
begin
  Result := (FButtonGrid <> nil) and Assigned(FButtonGrid.DataSource) and
            Assigned(FButtonGrid.DataSource.DataSet);

end;

end.
