unit UCCDComboBox;

interface

uses
	Windows, Messages, Controls,
  StdCtrls, UCCDBaseComp, UCCDTypes;

type
	TCCDComboBox = class(TComboBox)
	private
		FListHandle: HWnd;
    FAboutCCD: TAboutCCD;
		FDefListProc: TFarProc;
		FHintWindow: THintWindow;
  private
		procedure CancelHint();
		procedure DisplayHint(Index: Integer; const HintText: String);
	protected
		procedure CloseUp(); override;
		procedure CreateWnd(); override;
		// Ocultar la intil propiedad ListHandle de TComboBox
		property ListHandle: HWnd read FListHandle;
		procedure ListWndProc(var Message: TMessage); virtual;
  published
    property AcercaCCD: TAboutCCD read FAboutCCD stored false;
	end;

implementation

uses
  Classes, Graphics;

{ TCCDComboBox }

{ Este mtodo se llama cada vez que se crea el control nativo de Windows.
	Aqu revisamos el estilo del combo para saber si debemos manejar o no
	la lista.
}
procedure TCCDComboBox.CreateWnd();
var
	comboBoxInfo: TComboBoxInfo;
begin
	inherited;
	if Style <> csSimple then begin
		{ Obtener el identificador de la lista del combo }
		comboBoxInfo.cbSize := SizeOf(comboBoxInfo);
		GetComboBoxInfo(Handle, comboBoxInfo);
		FListHandle := comboBoxInfo.hwndList;
		{ Obtener el manejador de ventana de la lista }
		FDefListProc := TFarProc(GetWindowLong(FListHandle, GWL_WNDPROC));
		{ Reemplazar el manejador de ventana de la lista }
		SetWindowLong(
			FListHandle,
			GWL_WNDPROC,
			LongInt(Classes.MakeObjectInstance(ListWndProc))
		);
	end;
end;

{ Procedimiento de ventana para manejar la lista del combo
}
procedure TCCDComboBox.ListWndProc(var Message: TMessage);
var
	index: SmallInt;
	listRect: TRect;
	hintText: String;
	cursorPos: TSmallPoint;
begin
	with Message do
	begin
		case Msg of
			WM_MOUSEMOVE:
			begin
				// Obtener coordenadas del cursor
				cursorPos := TWMMouseMove(Message).Pos;
				// Obtener el rectngulo que ocupa la lista
				Windows.GetClientRect(FListHandle, listRect);
				InflateRect(listRect, -2, 0);
				// Asegurarnos que el cursor est sobre la lista
				if PtInRect(listRect, SmallPointToPoint(cursorPos)) then
				begin
					// Obtener el tem marcado
					index := SendMessage(FListHandle, LB_ITEMFROMPOINT, 0, LongInt(cursorPos));
					hintText := Items[index];
					{ Mostrar el hint slo si es necesario
					  Debemos seleccionar la fuente del combo en su canvas
					  para obtener la anchura correcta del texto
          }
					Canvas.Font := Font;
					if Canvas.TextWidth(hintText) > (listRect.Right - listRect.Left)
						then DisplayHint(index, hintText)
						else CancelHint;
				end
				else
					CancelHint;
			end;
		end;
		// Llamar al manejador original de la lista
		Result := CallWindowProc(FDefListProc, FListHandle, Msg, WParam, LParam);
	end;
end;

{ Muestra el hint indicado en la posicin correspondiente al tem
	cuyo ndice se indica.
}
procedure TCCDComboBox.DisplayHint(Index: Integer; const HintText: String);
var
	hintRect: TRect;
	hintSize: TSize;
begin
	// Crear el hint si es necesario
	if not Assigned(FHintWindow) then begin
		FHintWindow := THintWindow.Create(Self);
		FHintWindow.Color := clInfoBk;
	end;
	// Usar la fuente del combo en el hint
	FHintWindow.Canvas.Font := Font;
	// Calcular la dimensiones del hint
	hintRect := FHintWindow.CalcHintRect(300, HintText, nil);
	hintSize.cx := hintRect.Right - hintRect.Left;
	hintSize.cy := hintRect.Bottom - hintRect.Top;
	// Obtener el rectngulo que ocupa el tem en coord. absolutas
	SendMessage(FListHandle, LB_GETITEMRECT, Index, LongInt(@hintRect));
	MapWindowPoints(FListHandle, HWND_DESKTOP, hintRect, 2);
	// Ajustar el ancho
	hintRect.Right := hintRect.Left + hintSize.cx;
	hintRect.Bottom := hintRect.Top + hintSize.cy;
	FHintWindow.ActivateHint(hintRect, HintText);
end;

{ Destruye la ventana del hint
}
procedure TCCDComboBox.CancelHint();
begin
	if Assigned(FHintWindow) then
  begin
		FHintWindow.ReleaseHandle();
		FHintWindow := nil;
	end;
end;

{ Al cerrar la lista debemos cancelar el hint
}
procedure TCCDComboBox.CloseUp();
begin
	inherited;
	CancelHint();
end;

end.
