unit UCCDCheckTreeView;

interface

uses
	Classes, Controls, ComCtrls,
  ImgList, UCCDBaseComp, UCCDTypes;

const
	// Posibles estados de un nodo
	csUnchecked = 1;
	csGrayed    = 2;
	csChecked   = 3;

type
	TCCDCheckTreeView = class(TTreeView)
	private
    FAboutCCD: TAboutCCD;
		FCheckBoxes: TImageList;
  private
		procedure LoadBitmaps();
		procedure UpdateNode(node: TTreeNode);
		procedure EquateChildren(node: TTreeNode);
		procedure SetStateImages(value: TCustomImageList);
	protected
		procedure CreateWnd(); override;
		procedure Added(node: TTreeNode); override;
		procedure Delete(node: TTreeNode); override;
		procedure MouseDown(Button: TMouseButton; Shift:
     TShiftState; X, Y: Integer); override;
	public
		constructor Create(AOwner: TComponent); override;
		procedure CheckNode(node: TTreeNode);
		procedure UncheckNode(node: TTreeNode);
	published
		property StateImages write SetStateImages;
    property AboutCCD: TAboutCCD read FAboutCCD stored false;
	end;

implementation

uses
	Graphics, CommCtrl;

{ TCCDCheckTreeView }

{$R TCCDCheckTreeView.res}

constructor TCCDCheckTreeView.Create(aOwner: TComponent);
begin
	inherited;
	FCheckBoxes := TImageList.Create(self);
	LoadBitmaps();
end;

{ Cargar las imgenes para las casillas
  Las imgenes se encuentran en el archivo TCCDCheckTreeView.res
  Puede sustituirse ste por otras imgenes o puede ampliarse la clase
  para que permita seleccionar en el inspector de objetos cul imagen usar
}
procedure TCCDCheckTreeView.LoadBitmaps();
var
	bitmap: TBitmap;
begin
	bitmap := TBitmap.Create();
	try
		bitmap.LoadFromResourceName(HInstance, 'UNCHECKED');
		FCheckBoxes.AddMasked(bitmap, clFuchsia);
		FCheckBoxes.AddMasked(bitmap, clFuchsia);
		bitmap.LoadFromResourceName(HInstance, 'GRAYED');
		FCheckBoxes.AddMasked(bitmap, clFuchsia);
		bitmap.LoadFromResourceName(HInstance, 'CHECKED');
		FCheckBoxes.AddMasked(bitmap, clFuchsia);
	finally
		bitmap.Free;
	end;
end;

procedure TCCDCheckTreeView.SetStateImages(value: TCustomImageList);
begin
	inherited StateImages := value;
	if not Assigned(StateImages) then
		TreeView_SetImageList(Handle, FCheckBoxes.Handle, TVSIL_STATE);
end;

procedure TCCDCheckTreeView.CreateWnd();
begin
	inherited;
	if not Assigned(StateImages) then
		TreeView_SetImageList(Handle, FCheckBoxes.Handle, TVSIL_STATE);
end;

procedure TCCDCheckTreeView.Added(node: TTreeNode);
begin
	inherited;
	if node.StateIndex = -1 then
		node.StateIndex := csUnchecked;
	if Assigned(node.Parent) then
		UpdateNode(node.Parent);
end;

procedure TCCDCheckTreeView.Delete(node: TTreeNode);
begin
	inherited;
	UpdateNode(node.Parent);
end;

{ Pone todos los nodos al mismo estado que su padre
  El mtodo se usa para marcar o desmarcar un nodo y todos sus subnodos
  directos e indirectos.
}
procedure TCCDCheckTreeView.EquateChildren(node: TTreeNode);
var
	child: TTreeNode;
begin
	Items.BeginUpdate();
	try
		child := node.GetFirstChild;
		while Assigned(child) do begin
			child.StateIndex := node.StateIndex;
			EquateChildren(child);
			child := child.getNextSibling();
		end;
	finally
		Items.EndUpdate();
	end;
end;

{ Actualiza el estado de la casilla de un nodo
  Primero se examinan los subnodos directos sumando el peso de cada uno:
  0, 1  2, segn el subnodo est desmarcado, sombreado o marcado.
  Si ningn subnodo est marcado, el peso es cero, si todos los subnodos
  estn marcados, el peso es el doble del nmero de subnodos. Cualquier
  otra suma corresponder a los casos de subnodos parcialmente marcados.
  Este algoritmo no es recursivo, sino que se basa en lo que indique cada
  subnodo directo. Despus se llama al mismo mtodo para el nodo padre.
}
procedure TCCDCheckTreeView.UpdateNode(node: TTreeNode);
var
	count: Integer;
	weight: Integer;
	child: TTreeNode;
begin
	if Assigned(node) then
	begin
		count := 0;
		weight := 0;
		child := node.GetFirstChild();
		while Assigned(child) do begin
			if not child.Deleting then begin
				Inc(weight, child.StateIndex - 1);
				Inc(count);
			end;
			child := child.GetNextSibling();
		end;
		if weight = 0 then
			node.StateIndex := csUnchecked
		else if weight < 2*count then
			node.StateIndex := csGrayed
		else
			node.StateIndex := csChecked;
		UpdateNode(node.Parent);
	end;
end;

procedure TCCDCheckTreeView.CheckNode(node: TTreeNode);
begin
	node.StateIndex := csChecked;
	EquateChildren(node);
	UpdateNode(node.Parent);
end;

procedure TCCDCheckTreeView.UncheckNode(node: TTreeNode);
begin
	node.StateIndex := csUnchecked;
	EquateChildren(node);
	UpdateNode(node.Parent);
end;

procedure TCCDCheckTreeView.MouseDown(Button:
 TMouseButton; Shift: TShiftState; X, Y: Integer);
var
	node: TTreeNode;
begin
	inherited;
	if (Button = mbLeft) and (htOnStateIcon in GetHitTestInfoAt(X, Y)) then
	begin
		node := GetNodeAt(X, Y);
		if node.StateIndex = csUnchecked
			then CheckNode(node)
			else UncheckNode(node);
	end;
end;

end.

