unit UCCDAutoCompleteEdits;

{******************************************************************************}
{  AutoCompleteEdit 1.1                                                        }
{  Autor:  Hctor Carrillo Randolph - Club Delphi                              }
{  Pgina de descargas: http://www.clubdelphi.com/componentes/                 }
{  Pas de orgen: Mxico, noviembre del 2005.                                 }
{                                                                              }
{  TCCDAutoCompleteEdit                                                           }
{                                                                              }
{  Sencillo componente que permite asociar un conjunto de palabras con         }
{  el campo de edicin. Estas palabras aparecern en una lista desplegable     }
{  segn se vayan encontrando coincidencias con el texto al estilo             }
{  autocompletar.                                                              }
{                                                                              }
{  TCCDDBAutoCompleteEdit (data-aware)                                            }
{                                                                              }
{  Tiene las mismas caractersticas que TAutoCompleteEdit y adems permite     }
{  ligar el componente con algn campo en un DataSet.                          }
{                                                                              }
{******************************************************************************}

interface

uses
  SysUtils, Classes, Controls, Windows, Messages, Graphics,
  StrUtils, Mask, StdCtrls, Forms, DB, DBCtrls, ExtCtrls,
  UCCDTypes;

type

  TCustomAutoCompleteEdit = class;

  TAutoCompleteList=class(TCustomListBox)
  private
    FCaseSensitive:Boolean;//Sensible a mayusculas/minusculas
    FDropDownCount:Integer;//Numero de renglones de la lista
    FDropDownWidth:Integer;//Ancho de la lista
    FSearchList:TStringList;//Lista de palabras
    FEditor: TCustomAutoCompleteEdit;//TEdit asociado con la lista
    FSaveText: String;//Almacena texto para poder deshacer cambios
    FOnDropDown: TNotifyEvent;//Se activa al desplegar la lista
    FOnCloseUp:TNotifyEvent;//Se activa al cerrar la lista
    procedure Accept;
    procedure Cancel;
    procedure CloseUp;
    procedure DoSearch(Texto:String);
    procedure DropDown;
    procedure ListClick(Sender: TObject);
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
    procedure KeyDown(var Key:Word; Shift: TShiftState);override;
    procedure KeyPress(var Key:Char);override;
    procedure CNDrawItem(var Message: TWMDrawItem);message CN_DRAWITEM;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy;override;
    property OnDropDown: TNotifyEvent read FOnDropDown write FOnDropDown;
    property OnCloseUp: TNotifyEvent read FOnCloseUp write FOnCloseUp;
  end;

  TCustomAutoCompleteEdit = class(TCustomMaskEdit)
  private
    FAboutCCD: TAboutCCD;
    FAutoComplete: Boolean;//Activa autocompletar
    FDelay:Cardinal;//Retardo antes de mostrar la lista
    FACList:TAutoCompleteList;//Lista desplegable asociada con el TEdit
    FTimer:TTimer;//Timer para controlar el retardo
    FUpdating:Boolean;//Bandera de actualizacion
  protected
    procedure CreateWnd;override;
    procedure CreateParams(var Params: TCreateParams);override;
    procedure Change;override;
    procedure KeyDown(var Key:Word; Shift: TShiftState);override;
    procedure KeyPress(var Key:Char);override;
    procedure WMKillFocus(var Message: TWMKillFocus);message WM_KILLFOCUS;
    procedure CMCancelMode(var Message: TCMCancelMode);message CM_CANCELMODE;
    procedure Timer(Sender:TObject);
    // Set's & Get's
    procedure SetSearchList(const Value: TStringList);
    procedure SetText(const Value: String);
    procedure SetDelay(Value: Cardinal);
    procedure SetListColor(Value:TColor);
    procedure SetDropDownCount(Value:Integer);
    procedure SetDropDownWidth(Value:Integer);
    procedure SetCaseSensitive(Value:Boolean);
    procedure SetOnDropDown(Event:TNotifyEvent);
    procedure SetOnCloseUp(Event:TNotifyEvent);
    function  GetSearchList:TStringList;
    function  GetText:String;
    function  GetListColor:TColor;
    function  GetDropDownCount:Integer;
    function  GetDropDownWidth:Integer;
    function  GetCaseSensitive:Boolean;
    function  GetOnDropDown:TNotifyEvent;
    function  GetOnCloseUp:TNotifyEvent;
    property  AboutCCD: TAboutCCD read FAboutCCD write FAboutCCD stored false;
  public
    constructor Create(AOwner:TComponent);override;
    destructor Destroy;override;
    property AutoComplete:Boolean read FAutoComplete write FAutoComplete
             default True;
    property Delay:Cardinal read FDelay write SetDelay;
    property Text:String read GetText write SetText;
    property DropDownCount:Integer read GetDropDownCount write SetDropDownCount
             default 8;
    property DropDownWidth:Integer read GetDropDownWidth write SetDropDownWidth
             default 0;
    property CaseSensitive: Boolean read GetCaseSensitive write SetCaseSensitive
             default False;
    property ListColor:TColor read GetListColor write SetListColor;
    property SearchList:TStringList read GetSearchList write SetSearchList;
    property OnDropDown: TNotifyEvent read GetOnDropDown write SetOnDropDown;
    property OnCloseUp: TNotifyEvent read GetOnCloseUP write SetOnCloseUp;
  end;

  TCCDAutoCompleteEdit = class(TCustomAutoCompleteEdit)
  published
    property AboutCCD;
    property Anchors;
    property AutoComplete;
    property AutoSelect;
    property AutoSize;
    property BevelEdges;
    property BevelInner;
    property BevelKind;
    property BevelOuter;
    property BevelWidth;
    property BiDiMode;
    property BorderStyle;
    property CaseSensitive;
    property CharCase;
    property Color;
    property Ctl3D;
    property Cursor;
    property Delay;
    property DragCursor;
    property DragKind;
    property DragMode;
    property DropDownCount;
    property DropDownWidth;
    property Enabled;
    property Font;
    property HideSelection;
    property ImeMode;
    property ImeName;
    property ListColor;
    property MaxLength;
    property OEMConvert;
    property ParentBiDiMode;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property SearchList;
    property ShowHint;
    property TabOrder;
    property Text;
    property Visible;
    property OnChange;
    property OnClick;
    property OnCloseUp;
    property OnContextPopup;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnDropDown;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDock;
    property OnStartDrag;
  end;

  TCCDDBAutoCompleteEdit = class(TCustomAutoCompleteEdit)
  private
    FDataLink: TFieldDataLink;
    FAlignment: TAlignment;
    FCanvas: TControlCanvas;
    FFocused: Boolean;
    procedure DataChange(Sender: TObject);
    procedure ActiveChange(Sender: TObject);
    procedure EditingChange(Sender: TObject);
    function GetDataField: string;
    function GetDataSource: TDataSource;
    function GetField: TField;
    function GetReadOnly: Boolean;
    function GetTextMargins: TPoint;
    procedure ResetMaxLength;
    procedure UpdateData(Sender: TObject);
    procedure WMCut(var Message: TMessage); message WM_CUT;
    procedure WMPaste(var Message: TMessage); message WM_PASTE;
    procedure WMUndo(var Message: TMessage); message WM_UNDO;
    procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
    procedure CMEnter(var Message: TCMEnter); message CM_ENTER;
    procedure CMExit(var Message: TCMExit); message CM_EXIT;
    procedure CMGetDataLink(var Message: TMessage); message CM_GETDATALINK;
    // Set's & Get's
    procedure SetDataField(const Value: string);
    procedure SetDataSource(Value: TDataSource);
    procedure SetFocused(Value: Boolean);
    procedure SetReadOnly(Value: Boolean);
  protected
    procedure Change; override;
    function EditCanModify: Boolean; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure Reset; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Field: TField read GetField;    
  published
    property AboutCCD;
    property Anchors;
    property AutoComplete;
    property AutoSelect;
    property BevelEdges;
    property BevelInner;
    property BevelKind;
    property BevelOuter;
    property BiDiMode;
    property BorderStyle;
    property CaseSensitive;
    property CharCase;
    property Color;
    property Ctl3D;
    property Cursor;
    property DataField: string read GetDataField write SetDataField;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property Delay;
    property DragCursor;
    property DragKind;
    property DragMode;
    property DropDownCount;
    property DropDownWidth;
    property Enabled;
    property Font;
    property HideSelection;
    property ImeMode;
    property ImeName;
    property ListColor;
    property MaxLength;
    property OEMConvert;
    property ParentBiDiMode;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly: Boolean read GetReadOnly write SetReadOnly default False;
    property SearchList;
    property ShowHint;
    property TabOrder;
    property Text;
    property Visible;
    property OnChange;
    property OnClick;
    property OnCloseUp;
    property OnContextPopup;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnDropDown;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDock;
    property OnStartDrag;
  end;

implementation

{*********************************************************}
{ TCustomAutoCompleteEdit                                 }
{*********************************************************}
constructor TCustomAutoCompleteEdit.Create(AOwner:TComponent);
begin
  inherited Create(AOwner);
  ControlStyle:=ControlStyle+[csFixedHeight];
  FACList:=TAutoCompleteList.Create(Self);
  FACList.Parent:=Self;
  FACList.Visible:=False;
  FTimer:=TTimer.Create(Self);
  FTimer.Interval:=FDelay;
  FTimer.Enabled:=False;
  FTimer.OnTimer:=Timer;
  FUpDating:=False;
  FAutoComplete:=True;
  FDelay:=400;
  AutoSize:=True;
  CaseSensitive:=False;
  DropDownCount:=8;
  DropDownWidth:=0;
end;

destructor TCustomAutoCompleteEdit.Destroy;
begin
  FTimer.Free;
  FACList.Free;
  inherited Destroy;
end;

procedure TCustomAutoCompleteEdit.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  Params.Style := Params.Style or WS_CLIPCHILDREN;
end;

procedure TCustomAutoCompleteEdit.CreateWnd;
begin
  inherited CreateWnd;
  FACList.HandleNeeded;
end;

procedure TCustomAutoCompleteEdit.Change;
begin
  inherited Change;
  if (FAutoComplete) and not FUpdating then
  begin
    //Reiniciar contador
    FTimer.Enabled:=False;
    FTimer.Enabled:=True;
  end;
end;

procedure TCustomAutoCompleteEdit.KeyDown(var Key:Word; Shift: TShiftState);
begin
  if not FAutoComplete then Exit;
  if (FACList.Visible) and ( Key in [VK_UP, VK_DOWN]) then
  begin
    FACList.KeyDown(Key,Shift);
    Key:=VK_END;
  end
  else
  if Key in [VK_UP,VK_DOWN] then
  with FACList do
  begin
      FTimer.Enabled:=False;
      if Self.Text='' then
        Items.Assign(FSearchList)
      else
        DoSearch(Self.Text);
      if Count>0 then DropDown;
      if Key=VK_DOWN then
        ItemIndex:=0
      else
        ItemIndex:=Count-1;
      Key:=VK_END;
  end;
  inherited;
end;

procedure TCustomAutoCompleteEdit.KeyPress(var Key:Char);
begin
  inherited KeyPress(Key);
  if not FAutoComplete then Exit;
  if (FACList.Visible) and (Key in [#13,#27]) then
  begin
    FACList.KeyPress(Key);
    Key:=#0;
  end;
end;

procedure TCustomAutoCompleteEdit.WMKillFocus(var Message: TWMKillFocus);
begin
  FACList.Cancel;
  FACList.CloseUp;
  inherited;
end;

procedure TCustomAutoCompleteEdit.CMCancelMode(var Message: TCMCancelMode);
begin
  with Message do
    if (Sender <> Self) and (Sender <> FACList) then
    begin
      FACList.Cancel;
      FACList.CloseUp;
    end;
end;

procedure TCustomAutoCompleteEdit.Timer(Sender:TObject);
begin
  FTimer.Enabled:=False;
  if Text='' then
      FACList.CloseUp
  else
  with FACList do
  begin
    DoSearch(Self.Text);
    if Count>0 then
      DropDown
    else
      CloseUp;
  end;
end;

// Set's & Get's

procedure TCustomAutoCompleteEdit.SetText(const Value: String);
begin
  // Hay que evitar que se active autocompletar
  // cuando se actualiza el texto por cdigo y no por teclado
  FUpdating:=True;
  inherited Text:=Value;
  FUpdating:=False;
end;

procedure TCustomAutoCompleteEdit.SetDropDownCount(Value: Integer);
begin
  FACList.FDropDownCount:=Value;
end;

procedure TCustomAutoCompleteEdit.SetDropDownWidth(Value: Integer);
begin
  FACList.FDropDownWidth:=Value;
end;

procedure TCustomAutoCompleteEdit.SetCaseSensitive(Value: Boolean);
begin
  FACList.FCaseSensitive:=Value;
end;

procedure TCustomAutoCompleteEdit.SetSearchList(const Value: TStringList);
begin
  FACList.FSearchList.Assign(Value);
end;

procedure TCustomAutoCompleteEdit.SetDelay(Value: Cardinal);
begin
  FDelay:=Value;
  FTimer.Interval:=FDelay;
end;

procedure TCustomAutoCompleteEdit.SetListColor(Value: TColor);
begin
 FACList.Color:=Value;
end;

procedure TCustomAutoCompleteEdit.SetOnDropDown(Event:TNotifyEvent);
begin
  FACList.OnDropDown:=Event;
end;

procedure TCustomAutoCompleteEdit.SetOnCloseUp(Event:TNotifyEvent);
begin
  FACList.OnCloseUp:=Event;
end;

function TCustomAutoCompleteEdit.GetText:String;
begin
  Result:=(inherited Text);
end;

function TCustomAutoCompleteEdit.GetListColor:TColor;
begin
  Result:=FACList.Color;
end;

function TCustomAutoCompleteEdit.GetDropDownCount:Integer;
begin
  Result:=FACList.FDropDownCount;
end;

function TCustomAutoCompleteEdit.GetDropDownWidth:Integer;
begin
  Result:=FACList.FDropDownWidth;
end;

function TCustomAutoCompleteEdit.GetSearchList:TStringList;
begin
  Result:=FACList.FSearchList;
end;

function TCustomAutoCompleteEdit.GetCaseSensitive:Boolean;
begin
  Result:=FACList.FCaseSensitive;
end;

function TCustomAutoCompleteEdit.GetOnDropDown:TNotifyEvent;
begin
  Result:=FACList.FOnDropDown;
end;

function TCustomAutoCompleteEdit.GetOnCloseUp:TNotifyEvent;
begin
  Result:=FACList.FOnCloseUp;
end;


{*******************************************************}
{ TDBAutoCompleteEdit                                   }
{*******************************************************}

constructor TCCDDBAutoCompleteEdit.Create(AOwner:TComponent);
begin
  inherited Create(AOwner);
  inherited ReadOnly := True;
  ControlStyle := ControlStyle + [csReplicatable];
  FDataLink := TFieldDataLink.Create;
  FDataLink.Control := Self;
  FDataLink.OnDataChange := DataChange;
  FDataLink.OnEditingChange := EditingChange;
  FDataLink.OnUpdateData := UpdateData;
  FDataLink.OnActiveChange := ActiveChange;
end;

destructor TCCDDBAutoCompleteEdit.Destroy;
begin
  FDataLink.Free;
  FDataLink := nil;
  inherited Destroy;
end;

procedure TCCDDBAutoCompleteEdit.ResetMaxLength;
var
  F: TField;
begin
  if (MaxLength > 0) and Assigned(DataSource) and
      Assigned(DataSource.DataSet) then
  begin
    F := DataSource.DataSet.FindField(DataField);
    if Assigned(F) and (F.DataType in [ftString, ftWideString])
       and (F.Size = MaxLength) then
      MaxLength := 0;
  end;
end;

procedure TCCDDBAutoCompleteEdit.Loaded;
begin
  inherited Loaded;
  ResetMaxLength;
  if (csDesigning in ComponentState) then DataChange(Self);
end;

procedure TCCDDBAutoCompleteEdit.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
    (AComponent = DataSource) then DataSource := nil;
end;

procedure TCCDDBAutoCompleteEdit.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
  if (Key = VK_DELETE) or ((Key = VK_INSERT) and (ssShift in Shift)) then
    FDataLink.Edit;
end;

procedure TCCDDBAutoCompleteEdit.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);
  if (Key in [#32..#255]) and (FDataLink.Field <> nil) and
    not FDataLink.Field.IsValidChar(Key) then
  begin
    MessageBeep(0);
    Key := #0;
  end;
  case Key of
    ^H, ^V, ^X, #32..#255:
      FDataLink.Edit;
    #27:
      begin
        FDataLink.Reset;
        SelectAll;
        Key := #0;
      end;
  end;
end;

function TCCDDBAutoCompleteEdit.EditCanModify: Boolean;
begin
  Result := FDataLink.Edit;
end;

procedure TCCDDBAutoCompleteEdit.Reset;
begin
  FDataLink.Reset;
  SelectAll;
end;

procedure TCCDDBAutoCompleteEdit.SetFocused(Value: Boolean);
begin
  if FFocused <> Value then
  begin
    FFocused := Value;
    if (FAlignment <> taLeftJustify) and not IsMasked then Invalidate;
    FDataLink.Reset;
  end;
end;

procedure TCCDDBAutoCompleteEdit.Change;
begin
  FDataLink.Modified;
  inherited Change;
end;

function TCCDDBAutoCompleteEdit.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TCCDDBAutoCompleteEdit.SetDataSource(Value: TDataSource);
begin
  if not (FDataLink.DataSourceFixed and (csLoading in ComponentState)) then
    FDataLink.DataSource := Value;
  if Value <> nil then Value.FreeNotification(Self);
end;

function TCCDDBAutoCompleteEdit.GetDataField: string;
begin
  Result := FDataLink.FieldName;
end;

procedure TCCDDBAutoCompleteEdit.SetDataField(const Value: string);
begin
  if not (csDesigning in ComponentState) then
    ResetMaxLength;
  FDataLink.FieldName := Value;
end;

function TCCDDBAutoCompleteEdit.GetReadOnly: Boolean;
begin
  Result := FDataLink.ReadOnly;
end;

procedure TCCDDBAutoCompleteEdit.SetReadOnly(Value: Boolean);
begin
  FDataLink.ReadOnly := Value;
end;

function TCCDDBAutoCompleteEdit.GetField: TField;
begin
  Result := FDataLink.Field;
end;

procedure TCCDDBAutoCompleteEdit.ActiveChange(Sender: TObject);
begin
  ResetMaxLength;
end;

procedure TCCDDBAutoCompleteEdit.DataChange(Sender: TObject);
begin
  if FDataLink.Field <> nil then
  begin
    if FAlignment <> FDataLink.Field.Alignment then
    begin
      EditText := '';
      FAlignment := FDataLink.Field.Alignment;
    end;
    EditMask := FDataLink.Field.EditMask;
    if not (csDesigning in ComponentState) then
    begin
      if (FDataLink.Field.DataType in [ftString, ftWideString]) and
         (MaxLength = 0) then
        MaxLength := FDataLink.Field.Size;
    end;
    if FFocused and FDataLink.CanModify then
      Text := FDataLink.Field.Text
    else
    begin
      EditText := FDataLink.Field.DisplayText;
      if FDataLink.Editing then
        Modified := True;
    end;
  end else
  begin
    FAlignment := taLeftJustify;
    EditMask := '';
    if csDesigning in ComponentState then
      EditText := Name else
      EditText := '';
  end;
end;

procedure TCCDDBAutoCompleteEdit.EditingChange(Sender: TObject);
begin
  inherited ReadOnly := not FDataLink.Editing;
end;

procedure TCCDDBAutoCompleteEdit.UpdateData(Sender: TObject);
begin
  ValidateEdit;
  FDataLink.Field.Text := Text;
end;

procedure TCCDDBAutoCompleteEdit.WMUndo(var Message: TMessage);
begin
  FDataLink.Edit;
  inherited;
end;

procedure TCCDDBAutoCompleteEdit.WMPaste(var Message: TMessage);
begin
  FDataLink.Edit;
  inherited;
end;

procedure TCCDDBAutoCompleteEdit.WMCut(var Message: TMessage);
begin
  FDataLink.Edit;
  inherited;
end;

procedure TCCDDBAutoCompleteEdit.CMEnter(var Message: TCMEnter);
begin
  SetFocused(True);
  inherited;
  if SysLocale.FarEast and FDataLink.CanModify then
    inherited ReadOnly := False;
end;

procedure TCCDDBAutoCompleteEdit.CMExit(var Message: TCMExit);
begin
  try
    FDataLink.UpdateRecord;
  except
    SelectAll;
    SetFocus;
    raise;
  end;
  SetFocused(False);
  CheckCursor;
  DoExit;
end;

procedure TCCDDBAutoCompleteEdit.CMGetDataLink(var Message: TMessage);
begin
  Message.Result := Integer(FDataLink);
end;

procedure TCCDDBAutoCompleteEdit.WMPaint(var Message: TWMPaint);
const
  AlignStyle : array[Boolean, TAlignment] of DWORD =
   ((WS_EX_LEFT, WS_EX_RIGHT, WS_EX_LEFT),
    (WS_EX_RIGHT, WS_EX_LEFT, WS_EX_LEFT));
var
  Left: Integer;
  Margins: TPoint;
  R: TRect;
  DC: HDC;
  PS: TPaintStruct;
  S: string;
  AAlignment: TAlignment;
  ExStyle: DWORD;
begin
  AAlignment := FAlignment;
  if UseRightToLeftAlignment then ChangeBiDiModeAlignment(AAlignment);
  if ((AAlignment = taLeftJustify) or FFocused) and
    not (csPaintCopy in ControlState) then
  begin
    if SysLocale.MiddleEast and HandleAllocated and (IsRightToLeft) then
    begin
      ExStyle := DWORD(GetWindowLong(Handle, GWL_EXSTYLE)) and
        (not WS_EX_RIGHT) and (not WS_EX_RTLREADING) and
        (not WS_EX_LEFTSCROLLBAR);
      if UseRightToLeftReading then ExStyle := ExStyle or WS_EX_RTLREADING;
      if UseRightToLeftScrollbar then ExStyle := ExStyle or WS_EX_LEFTSCROLLBAR;
      ExStyle := ExStyle or
        AlignStyle[UseRightToLeftAlignment, AAlignment];
      if DWORD(GetWindowLong(Handle, GWL_EXSTYLE)) <> ExStyle then
        SetWindowLong(Handle, GWL_EXSTYLE, ExStyle);
    end;
    inherited;
    Exit;
  end;
  if FCanvas = nil then
  begin
    FCanvas := TControlCanvas.Create;
    FCanvas.Control := Self;
  end;
  DC := Message.DC;
  if DC = 0 then DC := BeginPaint(Handle, PS);
  FCanvas.Handle := DC;
  try
    FCanvas.Font := Font;
    with FCanvas do
    begin
      R := ClientRect;
      if not (NewStyleControls and Ctl3D) and (BorderStyle = bsSingle) then
      begin
        Brush.Color := clWindowFrame;
        FrameRect(R);
        InflateRect(R, -1, -1);
      end;
      Brush.Color := Color;
      if not Enabled then
        Font.Color := clGrayText;
      if (csPaintCopy in ControlState) and (FDataLink.Field <> nil) then
      begin
        S := FDataLink.Field.DisplayText;
        case CharCase of
          ecUpperCase: S := AnsiUpperCase(S);
          ecLowerCase: S := AnsiLowerCase(S);
        end;
      end else
        S := EditText;
      if PasswordChar <> #0 then FillChar(S[1], Length(S), PasswordChar);
      Margins := GetTextMargins;
      case AAlignment of
        taLeftJustify: Left := Margins.X;
        taRightJustify: Left := ClientWidth - TextWidth(S) - Margins.X - 1;
      else
        Left := (ClientWidth - TextWidth(S)) div 2;
      end;
      if SysLocale.MiddleEast then UpdateTextFlags;
      TextRect(R, Left, Margins.Y, S);
    end;
  finally
    FCanvas.Handle := 0;
    if Message.DC = 0 then EndPaint(Handle, PS);
  end;
end;

function TCCDDBAutoCompleteEdit.GetTextMargins: TPoint;
var
  DC: HDC;
  SaveFont: HFont;
  I: Integer;
  SysMetrics, Metrics: TTextMetric;
begin
  if NewStyleControls then
  begin
    if BorderStyle = bsNone then I := 0 else
      if Ctl3D then I := 1 else I := 2;
    Result.X := SendMessage(Handle, EM_GETMARGINS, 0, 0) and $0000FFFF + I;
    Result.Y := I;
  end else
  begin
    if BorderStyle = bsNone then I := 0 else
    begin
      DC := GetDC(0);
      GetTextMetrics(DC, SysMetrics);
      SaveFont := SelectObject(DC, Font.Handle);
      GetTextMetrics(DC, Metrics);
      SelectObject(DC, SaveFont);
      ReleaseDC(0, DC);
      I := SysMetrics.tmHeight;
      if I > Metrics.tmHeight then I := Metrics.tmHeight;
      I := I div 4;
    end;
    Result.X := I;
    Result.Y := I;
  end;
end;


{*******************************************************}
{ TAutoCompleteList                                     }
{*******************************************************}
constructor TAutoCompleteList.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Style:=lbOwnerDrawFixed;
  FEditor:=TCustomAutoCompleteEdit(AOwner);
  ControlStyle:=ControlStyle+[csNoDesignVisible,csCaptureMouse];
  FSearchList:=TStringList.Create;
  Color:=clwindow;
  FCaseSensitive:=False;
  FDropDownWidth:=0;
  FDropDownCount:=8;
  OnClick:=ListClick;
  BorderStyle:=bsNone;
  ParentFont:=True;
  ParentColor:=True;
  Height:=21;
  TabStop := False;
end;

destructor TAutoCompleteList.Destroy;
begin
 FSearchList.Free;
 inherited;
end;


procedure TAutoCompleteList.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do
  begin
    WindowClass.Style :=  WindowClass.Style or CS_SAVEBITS;
    Style:=Style or WS_BORDER or WS_CLIPCHILDREN;
    {$IFDEF WIN32}
       ExStyle:=Params.ExStyle or WS_EX_TOOLWINDOW;
    {$ENDIF}
  end;
end;

procedure TAutoCompleteList.CreateWnd;
begin
  inherited CreateWnd;
  if not (csDesigning in ComponentState) then
  begin
    Windows.SetParent(Handle, 0);
    CallWindowProc(DefWndProc, Handle, WM_SETFOCUS, 0, 0);
  end;
end;

procedure TAutoCompleteList.KeyDown(var Key:Word; Shift: TShiftState);
begin
  if not Visible then Exit;
  if (Key = VK_UP) then
  begin
    if (ItemIndex=-1) and (Count>0) then
    begin
      ItemIndex:=Count-1;
      FEditor.Text:=Items[ItemIndex];
    end
    else
    if ItemIndex>=1 then
    begin
     ItemIndex:=ItemIndex-1;
     FEditor.Text:=Items[ItemIndex];
    end
    else
     Cancel;
  end
  else
  if (Key = VK_DOWN) then
  begin
    if (ItemIndex=-1) and (Count>0) then
    begin
      ItemIndex:=0;
      FEditor.Text:=Items[ItemIndex];
    end
    else
    if ItemIndex<=Count-2 then
    begin
     ItemIndex:=ItemIndex+1;
     FEditor.Text:=Items[ItemIndex];
    end
    else
     Cancel;
  end;
end;

procedure TAutoCompleteList.KeyPress(var Key:Char);
begin
  inherited KeyPress(Key);
  if Key = #13 then Accept;
  if Key = #27 then
  begin
    Cancel;
    CloseUp;
  end;
end;

procedure TAutoCompleteList.Accept;
begin
  if ItemIndex<>-1 then
  begin
    FEditor.Text:=Items[ItemIndex];
    CloseUp;
  end;
end;

procedure TAutoCompleteList.Cancel;
begin
  if not Visible then Exit;
  FEditor.Text:=FSaveText;
  ClearSelection;
end;

procedure TAutoCompleteList.ListClick(Sender: TObject);
begin
  Accept;
end;

procedure TAutoCompleteList.DropDown;
var
  P:TPoint;
  ListHeight,ListWidth,X,Y:Integer;
  SysBorderHeight,SysBorderWidth:Integer;
begin
  if (not Visible) and Assigned(FOnDropDown) then FOnDropDown(Self);
  FSaveText:=FEditor.Text;
  P:=FEditor.ClientOrigin;
  SysBorderWidth := GetSystemMetrics(SM_CXBORDER);
  SysBorderHeight := GetSystemMetrics(SM_CYBORDER);
  ListHeight:=Count*ItemHeight;
  if ListHeight=0 then
     ListHeight:=ItemHeight
  else
  if ListHeight>(ItemHeight*FDropDownCount) then
     ListHeight:=(ItemHeight*FDropDownCount);
  ListHeight:=ListHeight+(SysBorderHeight*2);
  Y:=P.Y+FEditor.ClientHeight+(3*SysBorderHeight);
  if Y+ListHeight>Screen.Height then
     Y:=P.Y-ListHeight;
  if FDropDownWidth<=FEditor.ClientWidth then
    ListWidth:=FEditor.Width
  else
    ListWidth:=FDropDownWidth;
  X:=P.X;
  Dec(Y,SysBorderHeight);
  Dec(X,SysBorderWidth*2);
  SetWindowPos(Handle, 0, X, Y, ListWidth, ListHeight,
               SWP_NOACTIVATE or SWP_SHOWWINDOW );
  Visible := True;
  ClearSelection;
  SetItemIndex(-1);
end;

procedure TAutoCompleteList.CNDrawItem(var Message: TWMDrawItem);
var
  State: TOwnerDrawState;
begin
  with Message.DrawItemStruct^ do
  begin
    State := TOwnerDrawState(LongRec(itemState).Lo);
    Canvas.Handle := hDC;
    Canvas.Font := Font;
    Canvas.Brush := Brush;
    if (Integer(itemID) >= 0) and (odSelected in State) then
    begin
      Canvas.Brush.Color := clHighlight;
      Canvas.Font.Color := clHighlightText
    end;
    if Integer(itemID) >= 0 then
      DrawItem(itemID, rcItem, State) else
      Canvas.FillRect(rcItem);
    Canvas.Handle := 0;
  end;
end;


procedure TAutoCompleteList.CloseUp;
begin
  if Assigned(FOnCloseUp) then FOnCloseUp(Self);
  Visible:=False;
end;

procedure TAutoCompleteList.DoSearch(Texto:String);

  function ComparaCadenas(S1,S2:String;Sensible:Boolean):Integer;
  begin
    if Sensible then
      Result:=AnsiCompareStr(S1,S2)
    else
      Result:=AnsiCompareText(S1,S2);
  end;

  function EsPrefijo(Cadena,Sub: String; Sensible: Boolean):Boolean;
  begin
    if Sensible then
      Result:=AnsiStartsStr(Sub,Cadena)
    else
      Result:=AnsiStartsText(Sub,Cadena);
  end;

  function BinarySearch(Cadena: String; Lista: TStringList;
                     Sensible: Boolean):Integer;
  var
    Pos,Inf,Mitad,Sup,Comp,Ancho:Integer;
  begin
     Pos:=-1;
     Sup:=0;
     Inf:=Lista.Count-1;
     Ancho:=Length(Cadena);
     while (Sup<=Inf) do
     begin
       Mitad:= (Inf + Sup) div 2;
       Comp:=ComparaCadenas(Cadena,Copy(Lista.Strings[Mitad],1,Ancho),Sensible);
       if Comp>0 then
         Sup:=Mitad+1 // Continua la bsqueda hacia abajo
       else
       if Comp<0 then
         Inf:=Mitad-1 // Continua la bsqueda hacia arriba
       else
       begin
         Pos:=Mitad;
         break;
       end;
     end; // fin de while (Sup<=Inf...
     Result:=Pos;
  end;

var
  I,Pos:Integer;
  Ocurrencia,Temp:String;
begin
  // La lista y el texto no deben ser vacios
  if (FSearchList.Count>0) and (Texto<>'')then
  begin
    Clear;
    // Comienza la bsqueda binaria para encontrar la primera coincidencia
    // Slo funciona con listas ordenadas
    if not FSearchList.Sorted then FSearchList.Sort;
    Pos:=BinarySearch(Texto,FSearchList,FCaseSensitive);
    if ( Pos<>-1 ) then
    begin
       if (Length(FSearchList[Pos])>Length(Texto)) then
       begin
          Items.Add(FSearchList[Pos]);
          Ocurrencia:=FSearchList[Pos];
       end;
       I:=Pos-1;
       Temp:=Ocurrencia;
       // Agregar coincidencias por encima del elemento
       while (I>=0) do
       begin
         if (FSearchList[I]<>Temp) and
            (Length(FSearchList[I])>Length(Texto)) then
         begin
            if EsPrefijo(FSearchList[I],Texto,FCaseSensitive) then
            begin
              Items.Insert(0,FSearchList[I]);
              Temp:=FSearchList[I];
            end
            else
              break;
         end;
         Dec(I);
       end;
       // Agregar coincidencias por debajo del elemento
       I:=Pos+1;
       Temp:=Ocurrencia;
       while (I<=FSearchList.Count-1) do
       begin
         if (FSearchList[I]<>Temp) and
            (Length(FSearchList[I])>Length(Texto)) then
         begin
            if EsPrefijo(FSearchList[I],Texto,FCaseSensitive) then
            begin
              Items.Add(FSearchList[I]);
              Temp:=FSearchList[I];
            end
            else
              break;
         end;
         Inc(I);
       end;
    end; // fin de (Pos<>-1...
  end; // fin de (Count>0...
end;

end.


